<?php
$page_title = "Seat Selection";
require_once '../includes/header.php';
require_once '../includes/auth.php';

// Require login
requireAuth();

// Get booking data from POST
$schedule_id = isset($_POST['schedule_id']) ? intval($_POST['schedule_id']) : 0;
$origin = isset($_POST['origin']) ? intval($_POST['origin']) : 0;
$destination = isset($_POST['destination']) ? intval($_POST['destination']) : 0;
$departure_date = isset($_POST['departure_date']) ? $_POST['departure_date'] : '';
$passengers = isset($_POST['passengers']) ? intval($_POST['passengers']) : 1;
$seat_class = isset($_POST['seat_class']) ? $_POST['seat_class'] : 'Ekonomi';

// Use departure_date as the travel date
$date = $departure_date;

// If no schedule_id but origin/destination provided, find schedule
if (!$schedule_id && $origin && $destination) {
    $schedule_query = "SELECT s.id FROM schedules s
                       WHERE s.origin_station_id = ? 
                       AND s.destination_station_id = ?
                       AND s.status = 'active'
                       LIMIT 1";
    $stmt = mysqli_prepare($conn, $schedule_query);
    mysqli_stmt_bind_param($stmt, "ii", $origin, $destination);
    mysqli_stmt_execute($stmt);
    $result = mysqli_stmt_get_result($stmt);
    $temp = mysqli_fetch_assoc($result);
    
    if ($temp) {
        $schedule_id = $temp['id'];
    }
}

// Redirect back if no required data
if (!$schedule_id || empty($date)) {
    $_SESSION['error'] = "Missing booking information. Please start from schedule page.";
    header("Location: schedule.php");
    exit();
}

// Fetch schedule details with price
$schedule_query = "SELECT s.*, t.train_number, t.train_name,
                   os.city as origin_city, os.station_name as origin_station,
                   ds.city as dest_city, ds.station_name as dest_station
                   FROM schedules s
                   JOIN trains t ON s.train_id = t.id
                   JOIN stations os ON s.origin_station_id = os.id
                   JOIN stations ds ON s.destination_station_id = ds.id
                   WHERE s.id = ?";
$stmt = mysqli_prepare($conn, $schedule_query);
mysqli_stmt_bind_param($stmt, "i", $schedule_id);
mysqli_stmt_execute($stmt);
$result = mysqli_stmt_get_result($stmt);
$schedule = mysqli_fetch_assoc($result);

if (!$schedule) {
    $_SESSION['error'] = "Schedule not found";
    header("Location: schedule.php");
    exit();
}

// Calculate price based on seat class
$base_price = floatval($schedule['price']);
$class_multiplier = 1.0;

switch($seat_class) {
    case 'Business':
    case 'Bisnis':
        $class_multiplier = 1.5;
        break;
    case 'Executive':
    case 'Eksekutif':
        $class_multiplier = 2.0;
        break;
    default: // Economy/Ekonomi
        $class_multiplier = 1.0;
}

$price_per_ticket = $base_price * $class_multiplier;
$subtotal = $price_per_ticket * $passengers;
$service_fee = 5000;
$total = $subtotal + $service_fee;

// Format class name for display
$display_class = $seat_class;
if ($seat_class == 'Ekonomi') $display_class = 'Economy';
if ($seat_class == 'Bisnis') $display_class = 'Business';
if ($seat_class == 'Eksekutif') $display_class = 'Executive';

require_once '../includes/navbar.php';
?>

<link rel="stylesheet" href="../assets/css/style.css">
<link rel="stylesheet" href="../assets/css/responsive.css">

<style>
.
</style>

<section class="seat-selection-section">
    <div class="container">
        <div class="seat-selection-container">
            
            <!-- Main Seat Selection Area -->
            <div class="seat-main-area">
                <h2 class="page-title">Seat Selection</h2>
                <p style="color: var(--gray); margin-bottom: 24px;">
                    <?php echo htmlspecialchars($schedule['train_name']); ?> - 
                    <?php echo htmlspecialchars($schedule['origin_city']); ?> → 
                    <?php echo htmlspecialchars($schedule['dest_city']); ?>
                </p>
                
                <!-- Seat Legend -->
                <div class="seat-legend">
                    <div class="legend-item">
                        <span class="legend-box available"></span>
                        <span>Available</span>
                    </div>
                    <div class="legend-item">
                        <span class="legend-box selected"></span>
                        <span>Selected</span>
                    </div>
                    <div class="legend-item">
                        <span class="legend-box occupied"></span>
                        <span>Occupied</span>
                    </div>
                </div>

                <!-- Seat Map -->
                <div class="seat-map-wrapper">
                    <div class="seat-map">
                        <?php
                        // Generate 10 rows of seats (A-D columns)
                        $occupied_seats = ['2B', '6A', '8C']; // Dummy occupied seats
                        
                        for ($row = 1; $row <= 10; $row++):
                        ?>
                            <div class="seat-row">
                                <?php foreach(['A', 'B'] as $col): 
                                    $seat_id = $row . $col;
                                    $is_occupied = in_array($seat_id, $occupied_seats);
                                ?>
                                    <button 
                                        type="button" 
                                        class="seat <?php echo $is_occupied ? 'occupied' : 'available'; ?>" 
                                        data-seat="<?php echo $seat_id; ?>"
                                        <?php echo $is_occupied ? 'disabled' : ''; ?>
                                    ><?php echo $seat_id; ?></button>
                                <?php endforeach; ?>
                                
                                <div class="seat-aisle"></div>
                                
                                <?php foreach(['C', 'D'] as $col): 
                                    $seat_id = $row . $col;
                                    $is_occupied = in_array($seat_id, $occupied_seats);
                                ?>
                                    <button 
                                        type="button" 
                                        class="seat <?php echo $is_occupied ? 'occupied' : 'available'; ?>" 
                                        data-seat="<?php echo $seat_id; ?>"
                                        <?php echo $is_occupied ? 'disabled' : ''; ?>
                                    ><?php echo $seat_id; ?></button>
                                <?php endforeach; ?>
                            </div>
                        <?php endfor; ?>
                    </div>
                </div>
            </div>

            <!-- Price Summary Sidebar -->
            <div class="price-sidebar">
                <div class="price-summary">
                    <h3>Price Summary</h3>
                    
                    <div class="summary-item">
                        <span class="summary-label"><?php echo $display_class; ?> Class</span>
                        <span class="summary-value"><?php echo $passengers; ?> × Rp <?php echo number_format($price_per_ticket, 0, ',', '.'); ?></span>
                    </div>
                    
                    <div class="summary-item">
                        <span class="summary-label">Service Fee</span>
                        <span class="summary-value">Rp <?php echo number_format($service_fee, 0, ',', '.'); ?></span>
                    </div>
                    
                    <div class="summary-divider"></div>
                    
                    <div class="summary-total">
                        <span class="total-label">Total</span>
                        <span class="total-amount">Rp <?php echo number_format($total, 0, ',', '.'); ?></span>
                    </div>
                </div>

                <!-- Action Buttons -->
                <form method="POST" action="confirm-booking.php" id="confirmForm">
                    <input type="hidden" name="schedule_id" value="<?php echo $schedule_id; ?>">
                    <input type="hidden" name="date" value="<?php echo htmlspecialchars($date); ?>">
                    <input type="hidden" name="passengers" value="<?php echo $passengers; ?>">
                    <input type="hidden" name="seat_class" value="<?php echo htmlspecialchars($seat_class); ?>">
                    <input type="hidden" name="selected_seats" id="selectedSeatsInput" value="">
                    
                    <div class="sidebar-actions">
                        <a href="booking.php" class="btn-back-sidebar">Back to Booking</a>
                        <button type="submit" class="btn-confirm-sidebar" id="confirmBtn">
                            Continue to Confirmation (<span id="selectedCount">0</span>/<?php echo $passengers; ?>)
                        </button>
                    </div>
                </form>
            </div>

        </div>
    </div>
</section>

<script>
// Seat selection logic
const seats = document.querySelectorAll('.seat:not(.occupied)');
const selectedSeatsInput = document.getElementById('selectedSeatsInput');
const selectedCountSpan = document.getElementById('selectedCount');
const confirmBtn = document.getElementById('confirmBtn');
const maxPassengers = <?php echo $passengers; ?>;
let selectedSeats = [];

seats.forEach(seat => {
    seat.addEventListener('click', function() {
        const seatNumber = this.getAttribute('data-seat');
        
        if (this.classList.contains('selected')) {
            // Deselect
            this.classList.remove('selected');
            this.classList.add('available');
            selectedSeats = selectedSeats.filter(s => s !== seatNumber);
        } else {
            // Select
            if (selectedSeats.length < maxPassengers) {
                this.classList.remove('available');
                this.classList.add('selected');
                selectedSeats.push(seatNumber);
            } else {
                alert(`You can only select ${maxPassengers} seat(s)`);
            }
        }
        
        updateSelectedSeats();
    });
});

function updateSelectedSeats() {
    selectedSeatsInput.value = selectedSeats.join(',');
    selectedCountSpan.textContent = selectedSeats.length;
    
    // Enable/disable confirm button
    if (selectedSeats.length === maxPassengers) {
        confirmBtn.disabled = false;
        confirmBtn.style.opacity = '1';
    } else {
        confirmBtn.disabled = true;
        confirmBtn.style.opacity = '0.5';
    }
}

// Form validation
document.getElementById('confirmForm').addEventListener('submit', function(e) {
    if (selectedSeats.length === 0) {
        e.preventDefault();
        alert('Please select at least one seat');
        return false;
    }
    
    if (selectedSeats.length !== maxPassengers) {
        e.preventDefault();
        alert(`Please select exactly ${maxPassengers} seat(s). Currently selected: ${selectedSeats.length}`);
        return false;
    }
});

// Initialize
updateSelectedSeats();
</script>

<?php require_once '../includes/footer.php'; ?>