<?php
require_once '../includes/config.php';

requireLogin();

$page_title = "Train Schedule";
require_once '../includes/header.php';
require_once '../includes/navbar.php';

// Fetch stations for dropdown
$stations_query = "SELECT * FROM stations ORDER BY city ASC";
$stations_result = mysqli_query($conn, $stations_query);
$stations = [];
while($station = mysqli_fetch_assoc($stations_result)) {
    $stations[] = $station;
}

// Handle search
$schedules = [];
$search_performed = false;
$from_id = isset($_GET['from']) ? intval($_GET['from']) : '';
$to_id = isset($_GET['to']) ? intval($_GET['to']) : '';
$date = isset($_GET['date']) ? $_GET['date'] : '';
$passengers = isset($_GET['passengers']) ? intval($_GET['passengers']) : 1;

if(!empty($from_id) && !empty($to_id)) {
    $search_performed = true;
    
    $schedule_query = "SELECT s.*, t.train_number, t.train_name, t.train_type,
                       os.city as origin_city, os.code as origin_code,
                       ds.city as dest_city, ds.code as dest_code
                       FROM schedules s
                       JOIN trains t ON s.train_id = t.id
                       JOIN stations os ON s.origin_station_id = os.id
                       JOIN stations ds ON s.destination_station_id = ds.id
                       WHERE s.origin_station_id = ? 
                       AND s.destination_station_id = ?
                       AND s.status = 'active'
                       ORDER BY s.departure_time ASC";
    
    $stmt = mysqli_prepare($conn, $schedule_query);
    mysqli_stmt_bind_param($stmt, "ii", $from_id, $to_id);
    mysqli_stmt_execute($stmt);
    $schedule_result = mysqli_stmt_get_result($stmt);
    
    while($row = mysqli_fetch_assoc($schedule_result)) {
        $schedules[] = $row;
    }
    
    // Get origin and destination names for display
    $origin_name = '';
    $dest_name = '';
    foreach($stations as $station) {
        if($station['id'] == $from_id) $origin_name = $station['city'];
        if($station['id'] == $to_id) $dest_name = $station['city'];
    }
}
?>

<link rel="stylesheet" href="../assets/css/style.css">

<section class="schedule-section">
    <div class="container">
        <!-- Page Header -->
        <div class="page-header">
            <div>
                <h1>Train Schedule</h1>
                <p>Find and book your perfect train journey</p>
            </div>
        </div>

        <!-- Search Form -->
        <div class="search-card">
            <form action="" method="GET" class="schedule-search-form">
                <div class="search-grid">
                    <div class="form-group">
                        <label for="from">Departure Station</label>
                        <select name="from" id="from" required>
                            <option value="">Select departure station</option>
                            <?php foreach($stations as $station): ?>
                                <option value="<?php echo $station['id']; ?>" 
                                    <?php echo ($from_id == $station['id']) ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($station['city']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <div class="form-group">
                        <label for="to">Arrival Station</label>
                        <select name="to" id="to" required>
                            <option value="">Select arrival station</option>
                            <?php foreach($stations as $station): ?>
                                <option value="<?php echo $station['id']; ?>"
                                    <?php echo ($to_id == $station['id']) ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($station['city']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <div class="form-group">
                        <label for="date">Departure Date</label>
                        <input type="date" name="date" id="date" 
                               min="<?php echo date('Y-m-d'); ?>"
                               value="<?php echo htmlspecialchars($date); ?>" required>
                    </div>

                    <div class="form-group">
                        <label for="passengers">Passengers</label>
                        <select name="passengers" id="passengers">
                            <?php for($i = 1; $i <= 6; $i++): ?>
                                <option value="<?php echo $i; ?>" 
                                    <?php echo ($passengers == $i) ? 'selected' : ''; ?>>
                                    <?php echo $i; ?> Passenger<?php echo ($i > 1) ? 's' : ''; ?>
                                </option>
                            <?php endfor; ?>
                        </select>
                    </div>
                </div>

                <button type="submit" class="btn-search">
                    <svg width="20" height="20" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                        <circle cx="11" cy="11" r="8" stroke="white" stroke-width="2"/>
                        <path d="M21 21l-4.35-4.35" stroke="white" stroke-width="2" stroke-linecap="round"/>
                    </svg>
                    Search Schedule
                </button>
            </form>
        </div>

        <!-- Results -->
        <?php if($search_performed): ?>
            <div class="results-section">
                <?php if(!empty($schedules)): ?>
                    <div class="results-header">
                        <div class="results-info">
                            <p>Showing <strong><?php echo count($schedules); ?> results</strong> for 
                                <strong><?php echo htmlspecialchars($origin_name); ?></strong> → 
                                <strong><?php echo htmlspecialchars($dest_name); ?></strong>
                            </p>
                        </div>
                        
                        <div class="sort-by">
                            <label for="sortBy">Sort by:</label>
                            <select id="sortBy" onchange="sortSchedules(this.value)">
                                <option value="departure_time">Departure Time</option>
                                <option value="price_low">Price: Low to High</option>
                                <option value="price_high">Price: High to Low</option>
                                <option value="duration">Duration</option>
                            </select>
                        </div>
                    </div>

                    <div class="schedule-table-container">
                        <table class="schedule-table">
                            <thead>
                                <tr>
                                    <th>Train Number</th>
                                    <th>Train Name</th>
                                    <th>Departure - Arrival</th>
                                    <th>Duration</th>
                                    <th>Price</th>
                                    <th>Action</th>
                                </tr>
                            </thead>
                            <tbody id="scheduleTableBody">
                                <?php foreach($schedules as $schedule): ?>
                                    <tr data-price="<?php echo $schedule['price']; ?>" data-duration="<?php echo $schedule['duration']; ?>">
                                        <td class="train-number"><?php echo htmlspecialchars($schedule['train_number']); ?></td>
                                        <td class="train-name"><?php echo htmlspecialchars($schedule['train_name']); ?></td>
                                        <td class="time-cell">
                                            <span class="time-main">
                                                <?php echo date('H:i', strtotime($schedule['departure_time'])); ?>
                                                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                    <path d="M5 12h14m-7-7l7 7-7 7" stroke="currentColor" stroke-width="2" stroke-linecap="round"/>
                                                </svg>
                                                <?php echo date('H:i', strtotime($schedule['arrival_time'])); ?>
                                            </span>
                                        </td>
                                        <td class="duration"><?php echo htmlspecialchars($schedule['duration']); ?></td>
                                        <td class="price">Rp <?php echo number_format($schedule['price'], 0, ',', '.'); ?></td>
                                        <td class="action-cell">
                                            <?php if($schedule['available_seats'] > 0): ?>
                                                <a href="booking.php?schedule_id=<?php echo $schedule['id']; ?>&date=<?php echo urlencode($date); ?>&passengers=<?php echo $passengers; ?>" 
                                                   class="btn-book">Book</a>
                                            <?php else: ?>
                                                <span class="btn-sold-out">Sold Out</span>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>

                    <div class="schedule-note">
                        <svg width="20" height="20" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                            <circle cx="12" cy="12" r="10" stroke="currentColor" stroke-width="2"/>
                            <path d="M12 16v-4m0-4h.01" stroke="currentColor" stroke-width="2" stroke-linecap="round"/>
                        </svg>
                        <p>Schedules may change due to operational conditions. Please check the latest information before departure.</p>
                    </div>
                <?php else: ?>
                    <div class="empty-state">
                        <svg width="80" height="80" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                            <circle cx="11" cy="11" r="8" stroke="currentColor" stroke-width="2"/>
                            <path d="M21 21l-4.35-4.35" stroke="currentColor" stroke-width="2" stroke-linecap="round"/>
                        </svg>
                        <h3>No Schedules Found</h3>
                        <p>Sorry, we couldn't find any trains for this route. Try different stations or dates.</p>
                    </div>
                <?php endif; ?>
            </div>
        <?php endif; ?>
    </div>
</section>

<script>
function sortSchedules(sortType) {
    const tbody = document.getElementById('scheduleTableBody');
    const rows = Array.from(tbody.querySelectorAll('tr'));
    
    rows.sort((a, b) => {
        switch(sortType) {
            case 'departure_time':
                const timeA = a.querySelector('.time-main').textContent.trim().split('→')[0].trim();
                const timeB = b.querySelector('.time-main').textContent.trim().split('→')[0].trim();
                return timeA.localeCompare(timeB);
                
            case 'price_low':
                const priceA = parseFloat(a.dataset.price);
                const priceB = parseFloat(b.dataset.price);
                return priceA - priceB;
                
            case 'price_high':
                const priceC = parseFloat(a.dataset.price);
                const priceD = parseFloat(b.dataset.price);
                return priceD - priceC;
                
            case 'duration':
                const durationA = a.dataset.duration;
                const durationB = b.dataset.duration;
                return durationA.localeCompare(durationB);
                
            default:
                return 0;
        }
    });
    
    tbody.innerHTML = '';
    rows.forEach(row => tbody.appendChild(row));
}
</script>

<?php require_once '../includes/footer.php'; ?>