<?php
$page_title = "My Tickets";
require_once '../includes/header.php';
require_once '../includes/auth.php';

// Require login
requireAuth();

// Fetch active tickets (upcoming bookings)
$active_query = "SELECT b.*, s.departure_time, s.arrival_time, s.price, t.train_name, t.train_number,
                 os.station_name as origin_name, os.city as origin_city,
                 ds.station_name as dest_name, ds.city as dest_city
                 FROM bookings b
                 JOIN schedules s ON b.schedule_id = s.id
                 JOIN trains t ON s.train_id = t.id
                 JOIN stations os ON s.origin_station_id = os.id
                 JOIN stations ds ON s.destination_station_id = ds.id
                 WHERE b.user_id = ? AND b.booking_status = 'upcoming'
                 ORDER BY b.travel_date ASC";
$stmt = mysqli_prepare($conn, $active_query);
mysqli_stmt_bind_param($stmt, "i", $_SESSION['user_id']);
mysqli_stmt_execute($stmt);
$active_result = mysqli_stmt_get_result($stmt);

// Fetch booking history (completed bookings)
$history_query = "SELECT b.*, s.departure_time, s.arrival_time, s.price, t.train_name, t.train_number,
                  os.station_name as origin_name, os.city as origin_city,
                  ds.station_name as dest_name, ds.city as dest_city
                  FROM bookings b
                  JOIN schedules s ON b.schedule_id = s.id
                  JOIN trains t ON s.train_id = t.id
                  JOIN stations os ON s.origin_station_id = os.id
                  JOIN stations ds ON s.destination_station_id = ds.id
                  WHERE b.user_id = ? 
                  AND (
                      b.booking_status = 'cancelled' 
                      OR b.travel_date < CURDATE()
                  )
                  ORDER BY b.travel_date DESC
                  LIMIT 10";
$stmt2 = mysqli_prepare($conn, $history_query);
mysqli_stmt_bind_param($stmt2, "i", $_SESSION['user_id']);
mysqli_stmt_execute($stmt2);
$history_result = mysqli_stmt_get_result($stmt2);

// Auto-update past bookings to 'completed' status
$update_query = "UPDATE bookings 
                 SET booking_status = 'completed' 
                 WHERE travel_date < CURDATE() 
                 AND booking_status = 'upcoming'";
mysqli_query($conn, $update_query);

// NOW load navbar after queries are ready
require_once '../includes/navbar.php';
?>

<link rel="stylesheet" href="../assets/css/style.css">
<link rel="stylesheet" href="../assets/css/responsive.css">

<section class="my-tickets-section">
    <div class="container">
        <!-- Page Header -->
        <div class="page-header">
            <div>
                <h1>My Tickets</h1>
                <p>Manage your current and past train bookings</p>
            </div>
            <a href="../pages/schedule.php" class="btn-book-new">
                <svg width="20" height="20" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                    <path d="M12 5v14m7-7H5" stroke="currentColor" stroke-width="2" stroke-linecap="round"/>
                </svg>
                Book New Ticket
            </a>
        </div>

        <!-- Active Tickets -->
        <div class="tickets-section">
            <h2>Active Tickets</h2>
            
            <?php if(mysqli_num_rows($active_result) > 0): ?>
                <div class="tickets-grid">
                    <?php while($ticket = mysqli_fetch_assoc($active_result)): ?>
                        <div class="ticket-card">
                            <div class="ticket-header">
                                <div class="ticket-info">
                                    <h3><?php echo htmlspecialchars($ticket['train_name']); ?> <?php echo htmlspecialchars($ticket['train_number']); ?></h3>
                                    <span class="ticket-code">Ticket #<?php echo htmlspecialchars($ticket['booking_code']); ?></span>
                                </div>
                                <span class="ticket-status status-<?php echo $ticket['payment_status']; ?>">
                                    <?php echo ucfirst($ticket['payment_status']); ?>
                                </span>
                            </div>

                            <div class="ticket-route">
                                <div class="route-point">
                                    <div class="route-icon">
                                        <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                            <circle cx="12" cy="12" r="10" stroke="currentColor" stroke-width="2"/>
                                        </svg>
                                    </div>
                                    <div class="route-details">
                                        <strong><?php echo htmlspecialchars($ticket['origin_city']); ?></strong>
                                        <span><?php echo date('H:i', strtotime($ticket['departure_time'])); ?></span>
                                    </div>
                                </div>

                                <div class="route-line"></div>

                                <div class="route-point">
                                    <div class="route-icon">
                                        <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                            <circle cx="12" cy="12" r="10" fill="currentColor"/>
                                        </svg>
                                    </div>
                                    <div class="route-details">
                                        <strong><?php echo htmlspecialchars($ticket['dest_city']); ?></strong>
                                        <span><?php echo date('H:i', strtotime($ticket['arrival_time'])); ?></span>
                                    </div>
                                </div>
                            </div>

                            <div class="ticket-details">
                                <div class="detail-item">
                                    <span class="detail-label">Date:</span>
                                    <span class="detail-value"><?php echo date('M d, Y', strtotime($ticket['travel_date'])); ?></span>
                                </div>
                                <div class="detail-item">
                                    <span class="detail-label">Seat:</span>
                                    <span class="detail-value"><?php echo htmlspecialchars($ticket['seat_number']); ?></span>
                                </div>
                            </div>

                            <div class="ticket-actions">
                                <button class="btn-view-booking" onclick='showBookingDetails(<?php echo json_encode($ticket); ?>)'>
                                    <svg width="18" height="18" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                        <path d="M15 12a3 3 0 11-6 0 3 3 0 016 0z" stroke="currentColor" stroke-width="2"/>
                                        <path d="M2.458 12C3.732 7.943 7.523 5 12 5c4.478 0 8.268 2.943 9.542 7-1.274 4.057-5.064 7-9.542 7-4.477 0-8.268-2.943-9.542-7z" stroke="currentColor" stroke-width="2"/>
                                    </svg>
                                    View Details
                                </button>
                            </div>
                        </div>
                    <?php endwhile; ?>
                </div>
            <?php else: ?>
                <div class="empty-state">
                    <svg width="80" height="80" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                        <rect x="3" y="8" width="18" height="12" rx="2" stroke="currentColor" stroke-width="2"/>
                        <path d="M7 8V6C7 4.89543 7.89543 4 9 4H15C16.1046 4 17 4.89543 17 6V8" stroke="currentColor" stroke-width="2"/>
                    </svg>
                    <h3>No Active Tickets</h3>
                    <p>You don't have any upcoming trips yet</p>
                    <a href="../pages/schedule.php" class="btn-primary">Book a Ticket</a>
                </div>
            <?php endif; ?>
        </div>

        <!-- Booking History -->
        <div class="tickets-section">
            <div class="section-header">
                <h2>Booking History</h2>
                <select class="filter-select" id="historyFilter">
                    <option value="3">Last 3 months</option>
                    <option value="6">Last 6 months</option>
                    <option value="12">Last year</option>
                </select>
            </div>

            <?php if(mysqli_num_rows($history_result) > 0): ?>
                <div class="history-list">
                    <?php while($history = mysqli_fetch_assoc($history_result)): ?>
                        <div class="history-item">
                            <div class="history-main">
                                <div class="history-info">
                                    <h3><?php echo htmlspecialchars($history['train_name']); ?> <?php echo htmlspecialchars($history['train_number']); ?></h3>
                                    <p class="history-route">
                                        <?php echo htmlspecialchars($history['origin_city']); ?> 
                                        <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                            <path d="M5 12h14m-7-7l7 7-7 7" stroke="currentColor" stroke-width="2" stroke-linecap="round"/>
                                        </svg>
                                        <?php echo htmlspecialchars($history['dest_city']); ?>
                                    </p>
                                    <p class="history-date">
                                        <?php echo date('M d, Y', strtotime($history['travel_date'])); ?> • 
                                        <?php echo date('H:i', strtotime($history['departure_time'])); ?> - 
                                        <?php echo date('H:i', strtotime($history['arrival_time'])); ?>
                                    </p>
                                    <p class="history-seat"><?php echo htmlspecialchars($history['seat_number']); ?></p>
                                </div>
                                <span class="history-status status-<?php echo $history['booking_status']; ?>">
                                    <?php echo ucfirst($history['booking_status']); ?>
                                </span>
                            </div>
                            <button class="btn-view-details" onclick='showBookingDetails(<?php echo json_encode($history); ?>)'>
                                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                    <path d="M9 5l7 7-7 7" stroke="currentColor" stroke-width="2" stroke-linecap="round"/>
                                </svg>
                            </button>
                        </div>
                    <?php endwhile; ?>
                </div>
            <?php else: ?>
                <div class="empty-state">
                    <p>No booking history available</p>
                </div>
            <?php endif; ?>
        </div>
    </div>
</section>

<!-- Booking Details Modal -->
<div id="bookingModal" class="modal">
    <div class="modal-content booking-modal-content">
        <span class="modal-close" onclick="closeBookingModal()">&times;</span>
        <h3>Booking Details</h3>
        
        <div id="bookingDetailsContainer" class="booking-details-content">
            <!-- Will be populated by JavaScript -->
        </div>
    </div>
</div>


<script>
function showBookingDetails(booking) {
    const modal = document.getElementById('bookingModal');
    const container = document.getElementById('bookingDetailsContainer');
    
    // Format date and time
    const travelDate = new Date(booking.travel_date);
    const formattedDate = travelDate.toLocaleDateString('en-US', { 
        weekday: 'long', 
        year: 'numeric', 
        month: 'long', 
        day: 'numeric' 
    });
    
    // Format price
    const formattedPrice = new Intl.NumberFormat('id-ID', {
        style: 'currency',
        currency: 'IDR',
        minimumFractionDigits: 0
    }).format(booking.total_price);
    
    const perTicketPrice = new Intl.NumberFormat('id-ID', {
        style: 'currency',
        currency: 'IDR',
        minimumFractionDigits: 0
    }).format(booking.price || (booking.total_price / booking.passengers));
    
    // Check if booking can be cancelled (24+ hours before departure)
    const travelDateTime = new Date(booking.travel_date + ' ' + booking.departure_time);
    const hoursUntil = (travelDateTime - new Date()) / (1000 * 60 * 60);
    
    // Can cancel if: payment is confirmed/paid, booking is upcoming, and 24+ hours before departure
    const isUpcoming = booking.booking_status === 'upcoming' || booking.payment_status === 'confirmed' || booking.payment_status === 'paid';
    const canCancel = isUpcoming && hoursUntil >= 24;
    
    container.innerHTML = `
        <div class="booking-route-visual">
            <div class="booking-route-point">
                <strong>${booking.origin_city}</strong>
                <span>${booking.departure_time.substring(0, 5)}</span>
            </div>
            <div class="booking-route-arrow">
                <svg width="32" height="32" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                    <path d="M5 12h14m-7-7l7 7-7 7" stroke="currentColor" stroke-width="2" stroke-linecap="round"/>
                </svg>
            </div>
            <div class="booking-route-point">
                <strong>${booking.dest_city}</strong>
                <span>${booking.arrival_time.substring(0, 5)}</span>
            </div>
        </div>
        
        <div class="booking-info-row">
            <span class="booking-info-label">Booking Code</span>
            <span class="booking-info-value">${booking.booking_code}</span>
        </div>
        
        <div class="booking-info-row">
            <span class="booking-info-label">Train</span>
            <span class="booking-info-value">${booking.train_name} (${booking.train_number})</span>
        </div>
        
        <div class="booking-info-row">
            <span class="booking-info-label">Travel Date</span>
            <span class="booking-info-value">${formattedDate}</span>
        </div>
        
        <div class="booking-info-row">
            <span class="booking-info-label">Seat Number</span>
            <span class="booking-info-value">${booking.seat_number}</span>
        </div>
        
        <div class="booking-info-row">
            <span class="booking-info-label">Number of Passengers</span>
            <span class="booking-info-value">${booking.passengers} Passenger${booking.passengers > 1 ? 's' : ''}</span>
        </div>
        
        <div class="booking-info-row">
            <span class="booking-info-label">Price per Ticket</span>
            <span class="booking-info-value">${perTicketPrice}</span>
        </div>
        
        <div class="booking-info-row">
            <span class="booking-info-label">Booking Status</span>
            <span class="booking-info-value">
                <span class="status-badge status-${booking.booking_status}">
                    ${booking.booking_status.charAt(0).toUpperCase() + booking.booking_status.slice(1)}
                </span>
            </span>
        </div>
        
        <div class="booking-info-row">
            <span class="booking-info-label">Payment Status</span>
            <span class="booking-info-value">
                <span class="status-badge status-${booking.payment_status}">
                    ${booking.payment_status.charAt(0).toUpperCase() + booking.payment_status.slice(1)}
                </span>
            </span>
        </div>
        
        <div class="booking-total">
            <span class="booking-total-label">Total Amount</span>
            <span class="booking-total-value">${formattedPrice}</span>
        </div>
        
        ${canCancel ? `
            <div class="modal-actions">
                <button class="btn-cancel-modal" onclick="confirmCancel(${booking.id}, '${booking.booking_code}')">
                    <svg width="18" height="18" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                        <path d="M18 6L6 18M6 6l12 12" stroke="currentColor" stroke-width="2" stroke-linecap="round"/>
                    </svg>
                    Cancel This Booking
                </button>
            </div>
        ` : ''}
    `;
    
    modal.style.display = 'block';
}

function closeBookingModal() {
    document.getElementById('bookingModal').style.display = 'none';
}

// Close modal when clicking outside
window.onclick = function(event) {
    const modal = document.getElementById('bookingModal');
    if (event.target == modal) {
        modal.style.display = 'none';
    }
}

function confirmCancel(bookingId, bookingCode) {
    if (confirm('Are you sure you want to cancel booking ' + bookingCode + '?\n\nRefund will be processed within 3-5 business days.')) {
        window.location.href = 'cancel-booking.php?id=' + bookingId;
    }
}
</script>

<?php require_once '../includes/footer.php'; ?>