<?php
$page_title = "Confirm Your Booking";
require_once '../includes/header.php';
require_once '../includes/auth.php';

// Require login
requireAuth();

// Get data from POST (from seat-selection.php)
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    header("Location: schedule.php");
    exit();
}

$schedule_id = isset($_POST['schedule_id']) ? intval($_POST['schedule_id']) : 0;
$travel_date = isset($_POST['travel_date']) ? $_POST['travel_date'] : (isset($_POST['date']) ? $_POST['date'] : '');
$passengers = isset($_POST['passengers']) ? intval($_POST['passengers']) : 1;
$seat_class = isset($_POST['seat_class']) ? $_POST['seat_class'] : 'Economy';
$selected_seats = isset($_POST['selected_seats']) ? $_POST['selected_seats'] : [];

// Handle selected_seats conversion from string to array
if (is_string($selected_seats) && !empty($selected_seats)) {
    $selected_seats = explode(',', $selected_seats);
    $selected_seats = array_filter($selected_seats);
    $selected_seats = array_map('trim', $selected_seats);
} elseif (!is_array($selected_seats)) {
    $selected_seats = [];
}

// Validate
if (!$schedule_id || !$travel_date || empty($selected_seats)) {
    $_SESSION['error'] = "Invalid booking data";
    header("Location: schedule.php");
    exit();
}

// Fetch schedule details
$schedule_query = "SELECT s.*, t.train_number, t.train_name,
                   os.city as origin_city, os.station_name as origin_station,
                   ds.city as dest_city, ds.station_name as dest_station
                   FROM schedules s
                   JOIN trains t ON s.train_id = t.id
                   JOIN stations os ON s.origin_station_id = os.id
                   JOIN stations ds ON s.destination_station_id = ds.id
                   WHERE s.id = ?";
$stmt = mysqli_prepare($conn, $schedule_query);
mysqli_stmt_bind_param($stmt, "i", $schedule_id);
mysqli_stmt_execute($stmt);
$result = mysqli_stmt_get_result($stmt);
$schedule = mysqli_fetch_assoc($result);

if (!$schedule) {
    $_SESSION['error'] = "Schedule not found";
    header("Location: schedule.php");
    exit();
}

// Calculate prices based on seat class (FIX: Case-insensitive & normalized)
$base_price = floatval($schedule['price']);
$class_multiplier = 1.0;

// Normalize seat class name (case-insensitive)
$seat_class_normalized = strtolower(trim($seat_class));

switch($seat_class_normalized) {
    case 'business':
    case 'bisnis':
        $class_multiplier = 1.5;
        $seat_class = 'Business'; // Standardize display name
        break;
    case 'executive':
    case 'eksekutif':
        $class_multiplier = 2.0;
        $seat_class = 'Executive'; // Standardize display name
        break;
    default: // Economy/Ekonomi
        $class_multiplier = 1.0;
        $seat_class = 'Economy'; // Standardize display name
}

$price_per_ticket = $base_price * $class_multiplier;
$subtotal = $price_per_ticket * $passengers;
$service_fee = 5000; // Fixed service fee per booking
$total = $subtotal + $service_fee;

// NOW load navbar
require_once '../includes/navbar.php';
?>

<link rel="stylesheet" href="../assets/css/style.css">
<link rel="stylesheet" href="../assets/css/responsive.css">

<section class="confirm-booking-section">
    <div class="container">
        <!-- Page Header -->
        <div class="page-header-center">
            <h1>Confirm Your Booking</h1>
            <p>Please review your ticket details before final confirmation.</p>
        </div>

        <div class="confirm-booking-grid">
            <!-- Left Column: Booking Summary -->
            <div class="booking-summary-card">
                <div class="card-header">
                    <svg width="20" height="20" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                        <rect x="3" y="8" width="18" height="12" rx="2" stroke="currentColor" stroke-width="2"/>
                        <path d="M7 8V6C7 4.89543 7.89543 4 9 4H15C16.1046 4 17 4.89543 17 6V8" stroke="currentColor" stroke-width="2"/>
                    </svg>
                    <h3>Booking Summary</h3>
                </div>

                <div class="summary-content">
                    <!-- Train Info -->
                    <div class="summary-train">
                        <h4><?php echo htmlspecialchars($schedule['train_name']); ?></h4>
                        <span class="train-code">Train Code: <?php echo htmlspecialchars($schedule['train_number']); ?></span>
                    </div>

                    <!-- Route Info -->
                    <div class="summary-route">
                        <div class="route-item">
                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                <circle cx="12" cy="12" r="3" stroke="currentColor" stroke-width="2"/>
                            </svg>
                            <div>
                                <strong><?php echo htmlspecialchars($schedule['origin_city']); ?></strong>
                                <span><?php echo htmlspecialchars($schedule['origin_station']); ?></span>
                            </div>
                        </div>

                        <svg width="20" height="20" class="route-arrow" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                            <path d="M5 12h14m-4-4l4 4-4 4" stroke="currentColor" stroke-width="2" stroke-linecap="round"/>
                        </svg>

                        <div class="route-item">
                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                <circle cx="12" cy="12" r="3" fill="currentColor"/>
                            </svg>
                            <div>
                                <strong><?php echo htmlspecialchars($schedule['dest_city']); ?></strong>
                                <span><?php echo htmlspecialchars($schedule['dest_station']); ?></span>
                            </div>
                        </div>
                    </div>

                    <!-- Trip Details -->
                    <div class="summary-details">
                        <div class="detail-row">
                            <svg width="18" height="18" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                <rect x="3" y="6" width="18" height="15" rx="2" stroke="currentColor" stroke-width="2"/>
                                <path d="M3 10h18M8 3v6m8-6v6" stroke="currentColor" stroke-width="2" stroke-linecap="round"/>
                            </svg>
                            <span class="detail-label"><?php echo date('F d, Y', strtotime($travel_date)); ?></span>
                        </div>

                        <div class="detail-row">
                            <svg width="18" height="18" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                <circle cx="12" cy="12" r="10" stroke="currentColor" stroke-width="2"/>
                                <path d="M12 6v6l4 2" stroke="currentColor" stroke-width="2" stroke-linecap="round"/>
                            </svg>
                            <span class="detail-label">
                                <?php echo date('H:i', strtotime($schedule['departure_time'])); ?> - 
                                <?php echo date('H:i', strtotime($schedule['arrival_time'])); ?>
                            </span>
                        </div>

                        <div class="detail-row">
                            <svg width="18" height="18" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                <path d="M17 21v-2a4 4 0 00-4-4H5a4 4 0 00-4 4v2M9 11a4 4 0 100-8 4 4 0 000 8z" stroke="currentColor" stroke-width="2"/>
                            </svg>
                            <span class="detail-label">Seat <?php echo implode(', ', $selected_seats); ?></span>
                        </div>

                        <div class="detail-row">
                            <svg width="18" height="18" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                <path d="M21 10c0 7-9 13-9 13s-9-6-9-13a9 9 0 0118 0z" stroke="currentColor" stroke-width="2"/>
                                <circle cx="12" cy="10" r="3" stroke="currentColor" stroke-width="2"/>
                            </svg>
                            <span class="detail-label"><?php echo htmlspecialchars($seat_class); ?> Class</span>
                        </div>
                    </div>

                    <!-- Passenger Count -->
                    <div class="summary-passengers">
                        <span><?php echo $passengers; ?> Passenger<?php echo ($passengers > 1) ? 's' : ''; ?></span>
                        <span class="passengers-price">Rp <?php echo number_format($subtotal, 0, ',', '.'); ?></span>
                    </div>
                </div>

                <!-- Passenger Information Section -->
                <div class="card-header">
                    <svg width="20" height="20" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                        <path d="M17 21v-2a4 4 0 00-4-4H5a4 4 0 00-4 4v2M9 11a4 4 0 100-8 4 4 0 000 8zM23 21v-2a4 4 0 00-3-3.87M16 3.13a4 4 0 010 7.75" stroke="currentColor" stroke-width="2"/>
                    </svg>
                    <h3>Passenger Information</h3>
                    <button class="btn-edit-icon" onclick="goBack()">
                        <svg width="18" height="18" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                            <path d="M11 4H4a2 2 0 00-2 2v14a2 2 0 002 2h14a2 2 0 002-2v-7M18.5 2.5a2.121 2.121 0 013 3L12 15l-4 1 1-4 9.5-9.5z" stroke="currentColor" stroke-width="2"/>
                        </svg>
                    </button>
                </div>

                <div class="passenger-list">
                    <?php 
                    // Simulate passenger data (in real app, this would come from booking form)
                    $passenger_names = [
                        ['name' => $_SESSION['full_name'], 'class' => $seat_class, 'seat' => $selected_seats[0]]
                    ];
                    
                    for($i = 1; $i < $passengers; $i++) {
                        $passenger_names[] = [
                            'name' => 'Passenger ' . ($i + 1),
                            'class' => $seat_class,
                            'seat' => isset($selected_seats[$i]) ? $selected_seats[$i] : 'Auto'
                        ];
                    }
                    
                    foreach($passenger_names as $index => $passenger): 
                    ?>
                        <div class="passenger-item">
                            <div class="passenger-avatar"><?php echo ($index + 1); ?></div>
                            <div class="passenger-info">
                                <strong><?php echo htmlspecialchars($passenger['name']); ?></strong>
                                <span><?php echo $passenger['class']; ?> Class - Seat <?php echo $passenger['seat']; ?></span>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>

            <!-- Right Column: Payment Details -->
            <div class="payment-details-card">
                <div class="card-header">
                    <svg width="20" height="20" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                        <rect x="1" y="4" width="22" height="16" rx="2" stroke="currentColor" stroke-width="2"/>
                        <path d="M1 10h22" stroke="currentColor" stroke-width="2"/>
                    </svg>
                    <h3>Payment Details</h3>
                </div>

                <div class="payment-content">
                    <!-- Price Breakdown -->
                    <div class="price-breakdown">
                        <div class="price-row">
                            <span>Subtotal</span>
                            <span>Rp <?php echo number_format($subtotal, 0, ',', '.'); ?></span>
                        </div>
                        <div class="price-row">
                            <span>Service Fee</span>
                            <span>Rp <?php echo number_format($service_fee, 0, ',', '.'); ?></span>
                        </div>
                        <div class="price-row total-row">
                            <strong>Total</strong>
                            <strong>Rp <?php echo number_format($total, 0, ',', '.'); ?></strong>
                        </div>
                    </div>

                    <!-- Payment Method (Disabled since no payment) -->
                    <div class="payment-method">
                        <label>Payment Method</label>
                        <select class="form-select" disabled>
                            <option>Credit Card</option>
                        </select>
                        <p class="payment-note">
                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                <circle cx="12" cy="12" r="10" stroke="currentColor" stroke-width="2"/>
                                <path d="M12 16v-4m0-4h.01" stroke="currentColor" stroke-width="2" stroke-linecap="round"/>
                            </svg>
                            All transactions are secured and encrypted.
                        </p>
                    </div>

                    <!-- Action Buttons -->
                    <form action="<?php echo BASE_URL; ?>pages/booking-process.php" method="POST">
                        <input type="hidden" name="schedule_id" value="<?php echo $schedule_id; ?>">
                        <input type="hidden" name="travel_date" value="<?php echo htmlspecialchars($travel_date); ?>">
                        <input type="hidden" name="passengers" value="<?php echo $passengers; ?>">
                        <input type="hidden" name="seat_class" value="<?php echo htmlspecialchars($seat_class); ?>">
                        <input type="hidden" name="total_price" value="<?php echo $total; ?>">
                        <?php foreach($selected_seats as $seat): ?>
                            <input type="hidden" name="selected_seats[]" value="<?php echo htmlspecialchars($seat); ?>">
                        <?php endforeach; ?>
                        
                        <button type="submit" class="btn-confirm">
                            <svg width="20" height="20" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                <path d="M22 11.08V12a10 10 0 11-5.93-9.14" stroke="currentColor" stroke-width="2"/>
                                <path d="M22 4L12 14.01l-3-3" stroke="currentColor" stroke-width="2" stroke-linecap="round"/>
                            </svg>
                            Confirm & Book Ticket
                        </button>
                        
                        <button type="button" class="btn-cancel" onclick="goBack()">
                            <svg width="20" height="20" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                <path d="M18 6L6 18M6 6l12 12" stroke="currentColor" stroke-width="2" stroke-linecap="round"/>
                            </svg>
                            Cancel
                        </button>
                    </form>
                </div>
            </div>
        </div>
    </div>
</section>

<style>
.confirm-booking-section {
    padding: 60px 0;
    background: var(--light-gray);
    min-height: 80vh;
}

.page-header-center {
    text-align: center;
    margin-bottom: 48px;
}

.page-header-center h1 {
    font-size: 36px;
    font-weight: 700;
    color: var(--dark);
    margin-bottom: 8px;
}

.page-header-center p {
    color: var(--gray);
    font-size: 16px;
}

.confirm-booking-grid {
    display: grid;
    grid-template-columns: 1.5fr 1fr;
    gap: 32px;
    max-width: 1200px;
    margin: 0 auto;
}

.booking-summary-card,
.payment-details-card {
    background: var(--white);
    border-radius: 16px;
    box-shadow: var(--shadow);
}

.card-header {
    display: flex;
    align-items: center;
    gap: 12px;
    padding: 24px;
    border-bottom: 2px solid var(--border);
    position: relative;
}

.card-header h3 {
    font-size: 20px;
    font-weight: 600;
    color: var(--dark);
    flex: 1;
}

.card-header svg {
    color: var(--primary);
}

.btn-edit-icon {
    background: none;
    border: none;
    color: var(--primary);
    cursor: pointer;
    padding: 8px;
    border-radius: 6px;
    transition: all 0.3s;
}

.btn-edit-icon:hover {
    background: var(--primary-light);
}

.summary-content {
    padding: 24px;
}

.summary-train {
    margin-bottom: 24px;
}

.summary-train h4 {
    font-size: 22px;
    font-weight: 600;
    color: var(--dark);
    margin-bottom: 4px;
}

.train-code {
    font-size: 14px;
    color: var(--gray);
}

.summary-route {
    display: flex;
    align-items: center;
    gap: 16px;
    padding: 20px;
    background: var(--light-gray);
    border-radius: 12px;
    margin-bottom: 24px;
}

.route-item {
    display: flex;
    align-items: center;
    gap: 12px;
    flex: 1;
}

.route-item svg {
    color: var(--primary);
    flex-shrink: 0;
}

.route-item strong {
    display: block;
    font-size: 16px;
    color: var(--dark);
}

.route-item span {
    font-size: 13px;
    color: var(--gray);
}

.route-arrow {
    color: var(--primary);
    flex-shrink: 0;
}

.summary-details {
    display: flex;
    flex-direction: column;
    gap: 16px;
    margin-bottom: 24px;
}

.detail-row {
    display: flex;
    align-items: center;
    gap: 12px;
}

.detail-row svg {
    color: var(--primary);
}

.detail-label {
    font-size: 15px;
    color: var(--dark);
    font-weight: 500;
}

.summary-passengers {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 16px 20px;
    background: var(--light-gray);
    border-radius: 8px;
    font-weight: 600;
}

.passengers-price {
    font-size: 18px;
    color: var(--primary);
}

.passenger-list {
    padding: 24px;
    display: flex;
    flex-direction: column;
    gap: 16px;
}

.passenger-item {
    display: flex;
    align-items: center;
    gap: 16px;
}

.passenger-avatar {
    width: 40px;
    height: 40px;
    border-radius: 50%;
    background: var(--primary-light);
    color: var(--primary);
    display: flex;
    align-items: center;
    justify-content: center;
    font-weight: 600;
}

.passenger-info strong {
    display: block;
    font-size: 15px;
    color: var(--dark);
    margin-bottom: 2px;
}

.passenger-info span {
    font-size: 13px;
    color: var(--gray);
}

.payment-content {
    padding: 24px;
}

.price-breakdown {
    margin-bottom: 24px;
}

.price-row {
    display: flex;
    justify-content: space-between;
    padding: 12px 0;
    border-bottom: 1px solid var(--border);
}

.price-row:last-child {
    border-bottom: none;
}

.total-row {
    padding-top: 16px;
    margin-top: 8px;
    border-top: 2px solid var(--border);
    font-size: 18px;
}

.payment-method {
    margin-bottom: 24px;
}

.payment-method label {
    display: block;
    font-weight: 600;
    color: var(--dark);
    margin-bottom: 8px;
}

.form-select {
    width: 100%;
    padding: 12px 16px;
    border: 2px solid var(--border);
    border-radius: 8px;
    font-size: 15px;
    background: var(--light-gray);
}

.payment-note {
    display: flex;
    align-items: flex-start;
    gap: 8px;
    margin-top: 12px;
    padding: 12px;
    background: var(--light-gray);
    border-radius: 8px;
    font-size: 13px;
    color: var(--gray);
}

.payment-note svg {
    color: var(--primary);
    flex-shrink: 0;
    margin-top: 2px;
}

.btn-confirm,
.btn-cancel {
    width: 100%;
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 8px;
    padding: 14px;
    border-radius: 8px;
    font-size: 16px;
    font-weight: 600;
    border: none;
    cursor: pointer;
    transition: all 0.3s;
}

.btn-confirm {
    background: var(--dark);
    color: var(--white);
    margin-bottom: 12px;
}

.btn-confirm:hover {
    background: #2d3748;
    transform: translateY(-2px);
}

.btn-cancel {
    background: var(--white);
    color: var(--dark);
    border: 2px solid var(--border);
}

.btn-cancel:hover {
    background: var(--light-gray);
}

@media (max-width: 1024px) {
    .confirm-booking-grid {
        grid-template-columns: 1fr;
    }
}
</style>

<script>
function goBack() {
    window.history.back();
}
</script>

<?php require_once '../includes/footer.php'; ?>