<?php
$page_title = "Book New Ticket";
require_once '../includes/header.php';
require_once '../includes/auth.php';

// Require login
requireAuth();

// Get schedule ID and other params from URL
$schedule_id = isset($_GET['schedule_id']) ? intval($_GET['schedule_id']) : 0;
$date = isset($_GET['date']) ? $_GET['date'] : date('Y-m-d', strtotime('+1 day'));
$passengers = isset($_GET['passengers']) ? intval($_GET['passengers']) : 1;

// Fetch schedule details if schedule_id is provided
$schedule = null;
if ($schedule_id > 0) {
    $schedule_query = "SELECT s.*, t.train_name, t.train_number as train_class,
                       os.id as origin_id, os.city as origin_city, os.station_name as origin_station,
                       ds.id as dest_id, ds.city as dest_city, ds.station_name as dest_station
                       FROM schedules s
                       JOIN trains t ON s.train_id = t.id
                       JOIN stations os ON s.origin_station_id = os.id
                       JOIN stations ds ON s.destination_station_id = ds.id
                       WHERE s.id = ? AND s.status = 'active'";
    $stmt = mysqli_prepare($conn, $schedule_query);
    mysqli_stmt_bind_param($stmt, "i", $schedule_id);
    mysqli_stmt_execute($stmt);
    $result = mysqli_stmt_get_result($stmt);
    $schedule = mysqli_fetch_assoc($result);
}

// Fetch all stations for dropdown
$stations_query = "SELECT * FROM stations WHERE status = 'active' ORDER BY city ASC";
$stations_result = mysqli_query($conn, $stations_query);
$stations = [];
while($station = mysqli_fetch_assoc($stations_result)) {
    $stations[] = $station;
}

require_once '../includes/navbar.php';
?>

<link rel="stylesheet" href="../assets/css/style.css">
<link rel="stylesheet" href="../assets/css/responsive.css">

<section class="booking-section">
    <div class="container">
        <!-- Page Header -->
        <div class="booking-header">
            <h1>Book New Ticket</h1>
            <p>Choose your route, date, and seat to book your train ticket.</p>
        </div>

        <div class="booking-container">
            <!-- Booking Form Card -->
            <div class="booking-form-card">
                <form id="bookingForm" method="POST" action="seat-selection.php">
                    <!-- Pass schedule_id if available -->
                    <input type="hidden" name="schedule_id" value="<?php echo $schedule_id; ?>">
                    <input type="hidden" name="date" value="<?php echo htmlspecialchars($date); ?>">
                    
                    <!-- Route Selection -->
                    <div class="form-section">
                        <h2 class="section-title">Route Selection</h2>
                        
                        <div class="route-grid">
                            <div class="form-group">
                                <label for="origin">From</label>
                                <select name="origin" id="origin" required <?php echo $schedule ? 'disabled' : ''; ?>>
                                    <option value="">Select departure station</option>
                                    <?php foreach($stations as $station): ?>
                                        <option value="<?php echo $station['id']; ?>"
                                            <?php echo ($schedule && $schedule['origin_id'] == $station['id']) ? 'selected' : ''; ?>>
                                            <?php echo htmlspecialchars($station['city']); ?> - <?php echo htmlspecialchars($station['station_name']); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                                <?php if($schedule): ?>
                                    <input type="hidden" name="origin" value="<?php echo $schedule['origin_id']; ?>">
                                <?php endif; ?>
                            </div>

                            <div class="route-swap">
                                <svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                    <path d="M7 16l-4-4m0 0l4-4m-4 4h18M17 8l4 4m0 0l-4 4m4-4H3" stroke="currentColor" stroke-width="2" stroke-linecap="round"/>
                                </svg>
                            </div>

                            <div class="form-group">
                                <label for="destination">To</label>
                                <select name="destination" id="destination" required <?php echo $schedule ? 'disabled' : ''; ?>>
                                    <option value="">Select destination station</option>
                                    <?php foreach($stations as $station): ?>
                                        <option value="<?php echo $station['id']; ?>"
                                            <?php echo ($schedule && $schedule['dest_id'] == $station['id']) ? 'selected' : ''; ?>>
                                            <?php echo htmlspecialchars($station['city']); ?> - <?php echo htmlspecialchars($station['station_name']); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                                <?php if($schedule): ?>
                                    <input type="hidden" name="destination" value="<?php echo $schedule['dest_id']; ?>">
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>

                    <!-- Date & Time -->
                    <div class="form-section">
                        <h2 class="section-title">Date & Time</h2>
                        
                        <div class="datetime-grid">
                            <div class="form-group">
                                <label for="departure_date">Departure Date</label>
                                <input type="date" name="departure_date" id="departure_date" 
                                       value="<?php echo htmlspecialchars($date); ?>"
                                       min="<?php echo date('Y-m-d'); ?>" required>
                            </div>

                            <div class="form-group">
                                <label for="preferred_time">Preferred Time</label>
                                <select name="preferred_time" id="preferred_time">
                                    <option value="">Any time</option>
                                    <option value="morning">Morning (06:00 - 12:00)</option>
                                    <option value="afternoon">Afternoon (12:00 - 18:00)</option>
                                    <option value="evening">Evening (18:00 - 00:00)</option>
                                </select>
                            </div>
                        </div>
                    </div>

                    <!-- Passenger Information -->
                    <div class="form-section">
                        <h2 class="section-title">Passenger Information</h2>
                        
                        <div class="form-group">
                            <label for="passengers">Number of Passengers</label>
                            <select name="passengers" id="passengers" required>
                                <?php for($i = 1; $i <= 6; $i++): ?>
                                    <option value="<?php echo $i; ?>" <?php echo ($passengers == $i) ? 'selected' : ''; ?>>
                                        <?php echo $i; ?> Passenger<?php echo ($i > 1) ? 's' : ''; ?>
                                    </option>
                                <?php endfor; ?>
                            </select>
                        </div>

                        <div class="seat-class-section">
                            <label class="section-label">Seat Class</label>
                            <div class="seat-class-options">
                                <label class="seat-class-option">
                                    <input type="radio" name="seat_class" value="Ekonomi" required>
                                    <span class="option-check"></span>
                                    <span class="option-text">Economy</span>
                                </label>
                                <label class="seat-class-option">
                                    <input type="radio" name="seat_class" value="Bisnis">
                                    <span class="option-check"></span>
                                    <span class="option-text">Business</span>
                                </label>
                                <label class="seat-class-option">
                                    <input type="radio" name="seat_class" value="Eksekutif" checked>
                                    <span class="option-check"></span>
                                    <span class="option-text">Executive</span>
                                </label>
                            </div>
                        </div>
                    </div>

                    <!-- Submit Button -->
                    <div class="form-actions">
                        <button type="submit" class="btn-next">
                            Continue to Seat Selection
                            <svg width="20" height="20" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                <path d="M5 12h14m-7-7l7 7-7 7" stroke="currentColor" stroke-width="2" stroke-linecap="round"/>
                            </svg>
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>
</section>

<script>
// Swap stations
document.querySelector('.route-swap').addEventListener('click', function() {
    const origin = document.getElementById('origin');
    const destination = document.getElementById('destination');
    
    const tempValue = origin.value;
    origin.value = destination.value;
    destination.value = tempValue;
});

// Form validation
document.getElementById('bookingForm').addEventListener('submit', function(e) {
    const origin = document.getElementById('origin').value;
    const destination = document.getElementById('destination').value;
    
    if (origin === destination && origin !== '') {
        e.preventDefault();
        alert('Origin and destination cannot be the same!');
        return false;
    }
});
</script>

<?php require_once '../includes/footer.php'; ?>