<?php
session_start();
require_once '../includes/config.php';
require_once '../includes/auth.php';

// Require authentication
requireAuth();

// Check if form is submitted
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    $_SESSION['error'] = "Invalid request method";
    header("Location: schedule.php");
    exit();
}

// Get form data from confirm-booking.php
$schedule_id = isset($_POST['schedule_id']) ? intval($_POST['schedule_id']) : 0;
$travel_date = isset($_POST['travel_date']) ? $_POST['travel_date'] : '';
$passengers = isset($_POST['passengers']) ? intval($_POST['passengers']) : 0;
$seat_class = isset($_POST['seat_class']) ? $_POST['seat_class'] : '';
$total_price = isset($_POST['total_price']) ? floatval($_POST['total_price']) : 0;
$selected_seats = isset($_POST['selected_seats']) ? $_POST['selected_seats'] : [];

// Handle selected_seats - it comes as an array from confirm-booking.php
if (!is_array($selected_seats)) {
    $selected_seats = [];
}

$errors = [];

if (!$schedule_id) {
    $errors[] = "Invalid schedule";
}

if (empty($travel_date)) {
    $errors[] = "Travel date is required";
}

if ($passengers < 1 || $passengers > 6) {
    $errors[] = "Invalid number of passengers";
}

if (empty($seat_class)) {
    $errors[] = "Seat class is required";
}

if (empty($selected_seats)) {
    $errors[] = "Please select at least one seat";
}

if (count($selected_seats) != $passengers) {
    $errors[] = "Number of selected seats (" . count($selected_seats) . ") must match number of passengers (" . $passengers . ")";
}

if ($total_price <= 0) {
    $errors[] = "Invalid price";
}

// If there are validation errors
if (!empty($errors)) {
    $_SESSION['error'] = implode("<br>", $errors);
    header("Location: schedule.php");
    exit();
}

// Verify schedule exists
$schedule_query = "SELECT s.*, t.train_name, t.train_number 
                   FROM schedules s 
                   JOIN trains t ON s.train_id = t.id 
                   WHERE s.id = ? AND s.status = 'active'";
$stmt = mysqli_prepare($conn, $schedule_query);
mysqli_stmt_bind_param($stmt, "i", $schedule_id);
mysqli_stmt_execute($stmt);
$result = mysqli_stmt_get_result($stmt);
$schedule = mysqli_fetch_assoc($result);

if (!$schedule) {
    $_SESSION['error'] = "Schedule not found or no longer available";
    header("Location: schedule.php");
    exit();
}

// Check if seats are still available
foreach ($selected_seats as $seat) {
    $seat_check_query = "SELECT id FROM bookings 
                         WHERE schedule_id = ? 
                         AND travel_date = ? 
                         AND seat_number = ? 
                         AND booking_status NOT IN ('cancelled')";
    $seat_stmt = mysqli_prepare($conn, $seat_check_query);
    mysqli_stmt_bind_param($seat_stmt, "iss", $schedule_id, $travel_date, $seat);
    mysqli_stmt_execute($seat_stmt);
    $seat_result = mysqli_stmt_get_result($seat_stmt);
    
    if (mysqli_num_rows($seat_result) > 0) {
        $_SESSION['error'] = "Seat " . htmlspecialchars($seat) . " is no longer available. Please select another seat.";
        header("Location: schedule.php");
        exit();
    }
}

// Generate unique booking code
$booking_code = 'ET' . date('Ymd') . strtoupper(substr(uniqid(), -6));

// Get user info
$user_id = $_SESSION['user_id'];
$passenger_name = $_SESSION['full_name'];
$passenger_email = $_SESSION['email'];
$phone = isset($_SESSION['phone']) ? $_SESSION['phone'] : '0000000000';

// Start transaction
mysqli_begin_transaction($conn);

try {
    // Insert ONE booking record with all seats
    $seats_string = implode(',', $selected_seats);
    
    $insert_query = "INSERT INTO bookings (
        booking_code, 
        user_id, 
        schedule_id, 
        travel_date, 
        passenger_name, 
        passenger_email, 
        passenger_phone, 
        seat_number, 
        passengers,
        total_price, 
        booking_status, 
        payment_status,
        created_at
    ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'upcoming', 'confirmed', NOW())";
    
    $booking_stmt = mysqli_prepare($conn, $insert_query);
    
    mysqli_stmt_bind_param(
        $booking_stmt, 
        "siisssssid",
        $booking_code,
        $user_id,
        $schedule_id,
        $travel_date,
        $passenger_name,
        $passenger_email,
        $phone,
        $seats_string,
        $passengers,
        $total_price
    );
    
    if (!mysqli_stmt_execute($booking_stmt)) {
        throw new Exception("Failed to create booking: " . mysqli_error($conn));
    }
    
    // Commit transaction
    mysqli_commit($conn);
    
    // Set success message with HTML formatting
    $_SESSION['success'] = "🎉 Booking confirmed successfully!<br>Your booking code is: <strong>" . $booking_code . "</strong><br>Please save this code for your reference.";
    
    // Redirect to My Tickets page
    header("Location: my-ticket.php");
    exit();
    
} catch (Exception $e) {
    // Rollback on error
    mysqli_rollback($conn);
    
    $_SESSION['error'] = "Booking failed: " . $e->getMessage() . "<br>Please try again or contact support.";
    header("Location: schedule.php");
    exit();
}
?>