<?php
/**
 * Authentication Middleware
 * File ini berisi fungsi-fungsi untuk handle authentication & authorization
 */

// Check if user is logged in
function checkAuth() {
    if (!isset($_SESSION['user_id'])) {
        return false;
    }
    return true;
}

// Check if user is admin
function checkAdminAuth() {
    if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
        return false;
    }
    return true;
}

// Require user to be logged in
function requireAuth() {
    if (!checkAuth()) {
        $_SESSION['error'] = "Please login to access this page";
        // Simpan path relatif dari root
        $current_page = str_replace(BASE_URL, '', 'https://' . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI']);
        $_SESSION['redirect_to'] = '../' . $current_page;
        header("Location: " . BASE_URL . "pages/login.php");
        exit();
    }
}

// Require user to be admin
function requireAdminAuth() {
    if (!checkAuth()) {
        $_SESSION['error'] = "Please login to access this page";
        header("Location: " . BASE_URL . "pages/login.php");
        exit();
    }
    
    if (!checkAdminAuth()) {
        $_SESSION['error'] = "Access denied. Admin only.";
        header("Location: " . BASE_URL . "index.php");
        exit();
    }
}

// Get current user data
function getCurrentUser($conn) {
    if (!checkAuth()) {
        return null;
    }
    
    $user_id = $_SESSION['user_id'];
    $query = "SELECT id, full_name, email, username, role FROM users WHERE id = ?";
    $stmt = mysqli_prepare($conn, $query);
    mysqli_stmt_bind_param($stmt, "i", $user_id);
    mysqli_stmt_execute($stmt);
    $result = mysqli_stmt_get_result($stmt);
    
    if ($row = mysqli_fetch_assoc($result)) {
        return $row;
    }
    
    return null;
}

// Verify password
function verifyPassword($password, $hash) {
    return password_verify($password, $hash);
}

// Hash password
function hashPassword($password) {
    return password_hash($password, PASSWORD_DEFAULT);
}

// Login user
function loginUser($conn, $identifier, $password) {
    // Check if identifier is email or username
    $query = "SELECT id, full_name, email, username, password, role FROM users WHERE email = ? OR username = ? LIMIT 1";
    $stmt = mysqli_prepare($conn, $query);
    mysqli_stmt_bind_param($stmt, "ss", $identifier, $identifier);
    mysqli_stmt_execute($stmt);
    $result = mysqli_stmt_get_result($stmt);
    
    if ($row = mysqli_fetch_assoc($result)) {
        // Verify password
        if (verifyPassword($password, $row['password'])) {
            // Set session
            $_SESSION['user_id'] = $row['id'];
            $_SESSION['username'] = $row['username'];
            $_SESSION['full_name'] = $row['full_name'];
            $_SESSION['email'] = $row['email'];
            $_SESSION['role'] = $row['role'];
            
            return [
                'success' => true,
                'message' => 'Login successful',
                'role' => $row['role']
            ];
        } else {
            return [
                'success' => false,
                'message' => 'Invalid password'
            ];
        }
    } else {
        return [
            'success' => false,
            'message' => 'User not found'
        ];
    }
}

// Register user
function registerUser($conn, $full_name, $email, $username, $password) {
    // Check if email already exists
    $query = "SELECT id FROM users WHERE email = ? OR username = ?";
    $stmt = mysqli_prepare($conn, $query);
    mysqli_stmt_bind_param($stmt, "ss", $email, $username);
    mysqli_stmt_execute($stmt);
    $result = mysqli_stmt_get_result($stmt);
    
    if (mysqli_num_rows($result) > 0) {
        return [
            'success' => false,
            'message' => 'Email or username already exists'
        ];
    }
    
    // Hash password
    $hashed_password = hashPassword($password);
    
    // Insert user
    $query = "INSERT INTO users (full_name, email, username, password, role) VALUES (?, ?, ?, ?, 'user')";
    $stmt = mysqli_prepare($conn, $query);
    mysqli_stmt_bind_param($stmt, "ssss", $full_name, $email, $username, $hashed_password);
    
    if (mysqli_stmt_execute($stmt)) {
        return [
            'success' => true,
            'message' => 'Registration successful'
        ];
    } else {
        return [
            'success' => false,
            'message' => 'Registration failed. Please try again.'
        ];
    }
}

// Logout user
function logoutUser() {
    // Unset all session variables
    $_SESSION = array();
    
    // Destroy session cookie
    if (isset($_COOKIE[session_name()])) {
        setcookie(session_name(), '', time()-3600, '/');
    }
    
    // Destroy session
    session_destroy();
    
    // Redirect to home
    header("Location: " . BASE_URL . "index.php");
    exit();
}

// Check if email is valid
function isValidEmail($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL);
}

// Sanitize input
function sanitizeInput($data) {
    $data = trim($data);
    $data = stripslashes($data);
    $data = htmlspecialchars($data);
    return $data;
}
?>