// ========================================
// Mobile Menu Toggle
// ========================================
const hamburger = document.getElementById('hamburger');
const navMenu = document.getElementById('navMenu');

if (hamburger && navMenu) {
    hamburger.addEventListener('click', () => {
        navMenu.classList.toggle('active');
        hamburger.classList.toggle('active');
        
        // Animate hamburger icon
        const spans = hamburger.querySelectorAll('span');
        if (navMenu.classList.contains('active')) {
            spans[0].style.transform = 'rotate(45deg) translate(5px, 5px)';
            spans[1].style.opacity = '0';
            spans[2].style.transform = 'rotate(-45deg) translate(7px, -6px)';
        } else {
            spans[0].style.transform = 'none';
            spans[1].style.opacity = '1';
            spans[2].style.transform = 'none';
        }
    });

    // Close menu when clicking outside
    document.addEventListener('click', (e) => {
        if (!hamburger.contains(e.target) && !navMenu.contains(e.target)) {
            navMenu.classList.remove('active');
            hamburger.classList.remove('active');
            const spans = hamburger.querySelectorAll('span');
            spans[0].style.transform = 'none';
            spans[1].style.opacity = '1';
            spans[2].style.transform = 'none';
        }
    });
}

// ========================================
// User Dropdown Menu
// ========================================
const userBtn = document.getElementById('userBtn');
const userDropdown = document.getElementById('userDropdown');

if (userBtn && userDropdown) {
    // Toggle dropdown on click
    userBtn.addEventListener('click', (e) => {
        e.stopPropagation();
        userDropdown.style.display = userDropdown.style.display === 'block' ? 'none' : 'block';
    });

    // Close dropdown when clicking outside
    document.addEventListener('click', (e) => {
        if (!userBtn.contains(e.target) && !userDropdown.contains(e.target)) {
            userDropdown.style.display = 'none';
        }
    });
}

// ========================================
// Form Validation
// ========================================
const searchForm = document.querySelector('.search-form');

if (searchForm) {
    searchForm.addEventListener('submit', (e) => {
        const from = document.getElementById('from');
        const to = document.getElementById('to');
        const date = document.getElementById('date');

        // Check if from and to are the same
        if (from.value === to.value) {
            e.preventDefault();
            alert('Departure and destination stations cannot be the same!');
            return false;
        }

        // Check if date is in the past
        const selectedDate = new Date(date.value);
        const today = new Date();
        today.setHours(0, 0, 0, 0);

        if (selectedDate < today) {
            e.preventDefault();
            alert('Please select a future date!');
            return false;
        }
    });
}

// ========================================
// Smooth Scroll
// ========================================
document.querySelectorAll('a[href^="#"]').forEach(anchor => {
    anchor.addEventListener('click', function (e) {
        const href = this.getAttribute('href');
        if (href !== '#' && document.querySelector(href)) {
            e.preventDefault();
            document.querySelector(href).scrollIntoView({
                behavior: 'smooth'
            });
        }
    });
});

// ========================================
// Auto-hide alerts/messages
// ========================================
const alerts = document.querySelectorAll('.alert, .success-message, .error-message');
alerts.forEach(alert => {
    setTimeout(() => {
        alert.style.transition = 'opacity 0.5s ease';
        alert.style.opacity = '0';
        setTimeout(() => alert.remove(), 500);
    }, 5000);
});

// ========================================
// Date Input - Set min date to today
// ========================================
const dateInputs = document.querySelectorAll('input[type="date"]');
const today = new Date().toISOString().split('T')[0];
dateInputs.forEach(input => {
    if (!input.hasAttribute('min')) {
        input.setAttribute('min', today);
    }
});

// ========================================
// Navbar scroll effect (optional)
// ========================================
let lastScroll = 0;
const navbar = document.querySelector('.navbar');

window.addEventListener('scroll', () => {
    const currentScroll = window.pageYOffset;
    
    if (currentScroll > 100) {
        navbar.style.boxShadow = '0 4px 12px rgba(0, 0, 0, 0.15)';
    } else {
        navbar.style.boxShadow = 'var(--shadow)';
    }
    
    lastScroll = currentScroll;
});

// ========================================
// Animation on scroll (optional)
// ========================================
const observerOptions = {
    threshold: 0.1,
    rootMargin: '0px 0px -50px 0px'
};

const observer = new IntersectionObserver((entries) => {
    entries.forEach(entry => {
        if (entry.isIntersecting) {
            entry.target.style.opacity = '1';
            entry.target.style.transform = 'translateY(0)';
        }
    });
}, observerOptions);

// Observe feature cards and promo cards
document.querySelectorAll('.feature-card, .promo-card').forEach(card => {
    card.style.opacity = '0';
    card.style.transform = 'translateY(20px)';
    card.style.transition = 'opacity 0.6s ease, transform 0.6s ease';
    observer.observe(card);
});

// ========================================
// Print/Download functionality (for ticket pages)
// ========================================
const printButtons = document.querySelectorAll('[data-action="print"]');
printButtons.forEach(btn => {
    btn.addEventListener('click', () => {
        window.print();
    });
});

// ========================================
// Loading state for buttons
// ========================================
document.querySelectorAll('form').forEach(form => {
    form.addEventListener('submit', function(e) {
        const submitBtn = this.querySelector('button[type="submit"]');
        if (submitBtn && !submitBtn.disabled) {
            const originalText = submitBtn.innerHTML;
            submitBtn.disabled = true;
            submitBtn.innerHTML = '<span>Loading...</span>';
            
            // Re-enable after 3 seconds as fallback
            setTimeout(() => {
                submitBtn.disabled = false;
                submitBtn.innerHTML = originalText;
            }, 3000);
        }
    });
});

// ========================================
// Password toggle: make show/hide work reliably and avoid double-toggle from inline onclick attributes
// ========================================
document.querySelectorAll('.password-toggle').forEach(btn => {
    // remove inline onclick attribute if present (prevents double toggle)
    if (btn.getAttribute('onclick')) {
        btn.removeAttribute('onclick');
        btn.onclick = null;
    }

    btn.addEventListener('click', function(e) {
        e.preventDefault();
        const wrapper = btn.closest('.input-icon');
        if (!wrapper) return;
        const input = wrapper.querySelector('input[type="password"], input[type="text"]');
        if (!input) return;
        const wasPwd = input.type === 'password';
        input.type = wasPwd ? 'text' : 'password';
        btn.setAttribute('aria-pressed', wasPwd ? 'true' : 'false');
    });
});