<?php
$page_title = "Manage Users";
include 'includes/admin-header.php';

// Handle role update
if(isset($_POST['update_role'])) {
    $user_id = (int)$_POST['user_id'];
    $new_role = mysqli_real_escape_string($conn, $_POST['role']);
    
    // Prevent admin from changing their own role
    if($user_id == $_SESSION['user_id']) {
        $_SESSION['error'] = "You cannot change your own role!";
    } else {
        $query = "UPDATE users SET role = ? WHERE id = ?";
        $stmt = mysqli_prepare($conn, $query);
        mysqli_stmt_bind_param($stmt, "si", $new_role, $user_id);
        
        if(mysqli_stmt_execute($stmt)) {
            $_SESSION['success'] = "User role updated successfully!";
        } else {
            $_SESSION['error'] = "Failed to update user role.";
        }
    }
    
    header("Location: users.php");
    exit();
}

// Handle delete
if(isset($_GET['delete'])) {
    $user_id = (int)$_GET['delete'];
    
    // Prevent admin from deleting themselves
    if($user_id == $_SESSION['user_id']) {
        $_SESSION['error'] = "You cannot delete your own account!";
    } else {
        $delete_query = "DELETE FROM users WHERE id = ?";
        $stmt = mysqli_prepare($conn, $delete_query);
        mysqli_stmt_bind_param($stmt, "i", $user_id);
        
        if(mysqli_stmt_execute($stmt)) {
            $_SESSION['success'] = "User deleted successfully!";
        } else {
            $_SESSION['error'] = "Cannot delete user. They may have associated bookings.";
        }
    }
    
    header("Location: users.php");
    exit();
}

// Get filter parameters
$role_filter = isset($_GET['role']) ? $_GET['role'] : '';
$search = isset($_GET['search']) ? mysqli_real_escape_string($conn, $_GET['search']) : '';

// Build query
$query = "SELECT u.*, 
          (SELECT COUNT(*) FROM bookings WHERE user_id = u.id) as total_bookings,
          (SELECT SUM(total_price) FROM bookings WHERE user_id = u.id AND payment_status = 'confirmed') as total_spent
          FROM users u WHERE 1=1";

if($role_filter) {
    $query .= " AND u.role = '$role_filter'";
}

if($search) {
    $query .= " AND (u.full_name LIKE '%$search%' OR u.email LIKE '%$search%' OR u.username LIKE '%$search%')";
}

$query .= " ORDER BY u.created_at DESC";
$users = mysqli_query($conn, $query);

// Get user statistics
$stats_query = "SELECT 
                (SELECT COUNT(*) FROM users WHERE role = 'user') as total_users,
                (SELECT COUNT(*) FROM users WHERE role = 'admin') as total_admins,
                (SELECT COUNT(*) FROM users WHERE created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)) as new_users_month
                ";
$stats_result = mysqli_query($conn, $stats_query);
$stats = mysqli_fetch_assoc($stats_result);

include 'includes/admin-sidebar.php';
?>

<div class="admin-container">
    <div class="page-header">
        <div>
            <h1>Manage Users</h1>
            <p>View and manage user accounts</p>
        </div>
    </div>

    <!-- Statistics -->
    <div class="stats-grid" style="margin-bottom: 32px;">
        <div class="stat-card">
            <div class="stat-icon">👥</div>
            <div class="stat-info">
                <h3><?php echo number_format($stats['total_users']); ?></h3>
                <p>Total Users</p>
            </div>
        </div>

        <div class="stat-card">
            <div class="stat-icon">👨‍💼</div>
            <div class="stat-info">
                <h3><?php echo number_format($stats['total_admins']); ?></h3>
                <p>Total Admins</p>
            </div>
        </div>

        <div class="stat-card">
            <div class="stat-icon">🆕</div>
            <div class="stat-info">
                <h3><?php echo number_format($stats['new_users_month']); ?></h3>
                <p>New Users (30 Days)</p>
            </div>
        </div>
    </div>

    <?php if(isset($_SESSION['success'])): ?>
        <div class="alert-success">
            <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <polyline points="20 6 9 17 4 12"></polyline>
            </svg>
            <?php echo $_SESSION['success']; unset($_SESSION['success']); ?>
        </div>
    <?php endif; ?>

    <?php if(isset($_SESSION['error'])): ?>
        <div class="alert-error">
            <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <circle cx="12" cy="12" r="10"></circle>
                <line x1="15" y1="9" x2="9" y2="15"></line>
                <line x1="9" y1="9" x2="15" y2="15"></line>
            </svg>
            <?php echo $_SESSION['error']; unset($_SESSION['error']); ?>
        </div>
    <?php endif; ?>

    <!-- Filters -->
    <div class="filters-bar">
        <form method="GET" class="filter-form">
            <div class="filter-group">
                <label>Search</label>
                <input type="text" name="search" placeholder="Search by name, email, or username..." 
                       value="<?php echo htmlspecialchars($search); ?>">
            </div>
            
            <div class="filter-group">
                <label>Role</label>
                <select name="role">
                    <option value="">All Roles</option>
                    <option value="user" <?php echo $role_filter == 'user' ? 'selected' : ''; ?>>User</option>
                    <option value="admin" <?php echo $role_filter == 'admin' ? 'selected' : ''; ?>>Admin</option>
                </select>
            </div>
            
            <button type="submit" class="btn btn-primary">
                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <circle cx="11" cy="11" r="8"></circle>
                    <path d="m21 21-4.35-4.35"></path>
                </svg>
                Filter
            </button>
            <a href="users.php" class="btn btn-outline">Reset</a>
        </form>
    </div>

    <!-- Users Table -->
    <div class="admin-card">
        <div class="table-container">
            <table class="data-table">
                <thead>
                    <tr>
                        <th>User</th>
                        <th>Username</th>
                        <th>Role</th>
                        <th>Total Bookings</th>
                        <th>Total Spent</th>
                        <th>Joined Date</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if(mysqli_num_rows($users) > 0): ?>
                        <?php while($user = mysqli_fetch_assoc($users)): ?>
                            <tr>
                                <td>
                                    <div style="display: flex; align-items: center; gap: 12px;">
                                        <div style="width: 40px; height: 40px; background: var(--primary); color: white; border-radius: 50%; display: flex; align-items: center; justify-content: center; font-weight: 700;">
                                            <?php echo strtoupper(substr($user['full_name'], 0, 1)); ?>
                                        </div>
                                        <div>
                                            <strong><?php echo htmlspecialchars($user['full_name']); ?></strong>
                                            <?php if($user['id'] == $_SESSION['user_id']): ?>
                                                <span style="color: var(--primary); font-size: 12px; font-weight: 600;"> (You)</span>
                                            <?php endif; ?>
                                            <br>
                                            <small style="color: var(--gray);"><?php echo htmlspecialchars($user['email']); ?></small>
                                        </div>
                                    </div>
                                </td>
                                <td><?php echo htmlspecialchars($user['username']); ?></td>
                                <td>
                                    <form method="POST" style="margin: 0;">
                                        <input type="hidden" name="user_id" value="<?php echo $user['id']; ?>">
                                        <select name="role" 
                                                class="status-badge status-<?php echo $user['role'] == 'admin' ? 'confirmed' : 'pending'; ?>" 
                                                style="border: none; cursor: pointer;"
                                                onchange="this.form.submit()"
                                                <?php echo $user['id'] == $_SESSION['user_id'] ? 'disabled' : ''; ?>>
                                            <option value="user" <?php echo $user['role'] == 'user' ? 'selected' : ''; ?>>User</option>
                                            <option value="admin" <?php echo $user['role'] == 'admin' ? 'selected' : ''; ?>>Admin</option>
                                        </select>
                                        <input type="hidden" name="update_role" value="1">
                                    </form>
                                </td>
                                <td><?php echo number_format($user['total_bookings']); ?> bookings</td>
                                <td>
                                    <?php if($user['total_spent']): ?>
                                        <strong>Rp <?php echo number_format($user['total_spent'], 0, ',', '.'); ?></strong>
                                    <?php else: ?>
                                        <span style="color: var(--gray);">-</span>
                                    <?php endif; ?>
                                </td>
                                <td><?php echo date('d M Y', strtotime($user['created_at'])); ?></td>
                                <td>
                                    <div class="table-actions">
                                        <?php if($user['id'] != $_SESSION['user_id']): ?>
                                            <a href="?delete=<?php echo $user['id']; ?>" 
                                               class="btn btn-danger btn-sm btn-icon"
                                               data-confirm="Are you sure you want to delete this user?"
                                               title="Delete">
                                                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                                    <polyline points="3 6 5 6 21 6"></polyline>
                                                    <path d="M19 6v14a2 2 0 0 1-2 2H7a2 2 0 0 1-2-2V6m3 0V4a2 2 0 0 1 2-2h4a2 2 0 0 1 2 2v2"></path>
                                                </svg>
                                            </a>
                                        <?php else: ?>
                                            <span style="color: var(--gray); font-size: 12px;">-</span>
                                        <?php endif; ?>
                                    </div>
                                </td>
                            </tr>
                        <?php endwhile; ?>
                    <?php else: ?>
                        <tr>
                            <td colspan="7" style="text-align: center; padding: 60px 20px;">
                                <div class="empty-state">
                                    <svg width="80" height="80" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                        <path d="M17 21v-2a4 4 0 0 0-4-4H5a4 4 0 0 0-4 4v2"></path>
                                        <circle cx="9" cy="7" r="4"></circle>
                                        <path d="M23 21v-2a4 4 0 0 0-3-3.87"></path>
                                        <path d="M16 3.13a4 4 0 0 1 0 7.75"></path>
                                    </svg>
                                    <h3>No users found</h3>
                                    <p>Try adjusting your filters or search terms</p>
                                </div>
                            </td>
                        </tr>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>

<?php include 'includes/admin-footer.php'; ?>