<?php
$page_title = "Manage Trains";
include 'includes/admin-header.php';

// Handle Add/Edit
if(isset($_POST['save_train'])) {
    $train_id = isset($_POST['train_id']) ? (int)$_POST['train_id'] : 0;
    $train_number = mysqli_real_escape_string($conn, $_POST['train_number']);
    $train_name = mysqli_real_escape_string($conn, $_POST['train_name']);
    $train_type = mysqli_real_escape_string($conn, $_POST['train_type']);
    $capacity = (int)$_POST['capacity'];
    
    if($train_id > 0) {
        // Update
        $query = "UPDATE trains SET train_number = ?, train_name = ?, train_type = ?, capacity = ? WHERE id = ?";
        $stmt = mysqli_prepare($conn, $query);
        mysqli_stmt_bind_param($stmt, "sssii", $train_number, $train_name, $train_type, $capacity, $train_id);
        $message = "Train updated successfully!";
    } else {
        // Insert
        $query = "INSERT INTO trains (train_number, train_name, train_type, capacity) VALUES (?, ?, ?, ?)";
        $stmt = mysqli_prepare($conn, $query);
        mysqli_stmt_bind_param($stmt, "sssi", $train_number, $train_name, $train_type, $capacity);
        $message = "Train added successfully!";
    }
    
    if(mysqli_stmt_execute($stmt)) {
        $_SESSION['success'] = $message;
    } else {
        $_SESSION['error'] = "Error: " . mysqli_error($conn);
    }
    
    header("Location: trains.php");
    exit();
}

// Handle Delete
if(isset($_GET['delete'])) {
    $train_id = (int)$_GET['delete'];
    
    $delete_query = "DELETE FROM trains WHERE id = ?";
    $stmt = mysqli_prepare($conn, $delete_query);
    mysqli_stmt_bind_param($stmt, "i", $train_id);
    
    if(mysqli_stmt_execute($stmt)) {
        $_SESSION['success'] = "Train deleted successfully!";
    } else {
        $_SESSION['error'] = "Cannot delete train. It may be associated with schedules.";
    }
    
    header("Location: trains.php");
    exit();
}

// Get train for editing
$edit_train = null;
if(isset($_GET['edit'])) {
    $train_id = (int)$_GET['edit'];
    $query = "SELECT * FROM trains WHERE id = ?";
    $stmt = mysqli_prepare($conn, $query);
    mysqli_stmt_bind_param($stmt, "i", $train_id);
    mysqli_stmt_execute($stmt);
    $result = mysqli_stmt_get_result($stmt);
    $edit_train = mysqli_fetch_assoc($result);
}

// Get all trains
$query = "SELECT * FROM trains ORDER BY train_number ASC";
$trains = mysqli_query($conn, $query);

include 'includes/admin-sidebar.php';
?>

<div class="admin-container">
    <div class="page-header">
        <div>
            <h1>Manage Trains</h1>
            <p>Add, edit, or remove trains from the system</p>
        </div>
        <button class="btn btn-primary" onclick="openModal()">
            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <line x1="12" y1="5" x2="12" y2="19"></line>
                <line x1="5" y1="12" x2="19" y2="12"></line>
            </svg>
            Add New Train
        </button>
    </div>

    <?php if(isset($_SESSION['success'])): ?>
        <div class="alert-success">
            <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <polyline points="20 6 9 17 4 12"></polyline>
            </svg>
            <?php echo $_SESSION['success']; unset($_SESSION['success']); ?>
        </div>
    <?php endif; ?>

    <?php if(isset($_SESSION['error'])): ?>
        <div class="alert-error">
            <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <circle cx="12" cy="12" r="10"></circle>
                <line x1="15" y1="9" x2="9" y2="15"></line>
                <line x1="9" y1="9" x2="15" y2="15"></line>
            </svg>
            <?php echo $_SESSION['error']; unset($_SESSION['error']); ?>
        </div>
    <?php endif; ?>

    <!-- Trains Table -->
    <div class="admin-card">
        <div class="table-container">
            <table class="data-table">
                <thead>
                    <tr>
                        <th>Train Number</th>
                        <th>Train Name</th>
                        <th>Type</th>
                        <th>Capacity</th>
                        <th>Created At</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if(mysqli_num_rows($trains) > 0): ?>
                        <?php while($train = mysqli_fetch_assoc($trains)): ?>
                            <tr>
                                <td><strong><?php echo htmlspecialchars($train['train_number']); ?></strong></td>
                                <td><?php echo htmlspecialchars($train['train_name']); ?></td>
                                <td>
                                    <span class="status-badge status-active">
                                        <?php echo $train['train_type']; ?>
                                    </span>
                                </td>
                                <td><?php echo $train['capacity']; ?> seats</td>
                                <td><?php echo date('d M Y', strtotime($train['created_at'])); ?></td>
                                <td>
                                    <div class="table-actions">
                                        <a href="?edit=<?php echo $train['id']; ?>" 
                                           class="btn btn-primary btn-sm btn-icon"
                                           onclick="editTrain(<?php echo htmlspecialchars(json_encode($train)); ?>); return false;"
                                           title="Edit">
                                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                                <path d="M11 4H4a2 2 0 0 0-2 2v14a2 2 0 0 0 2 2h14a2 2 0 0 0 2-2v-7"></path>
                                                <path d="M18.5 2.5a2.121 2.121 0 0 1 3 3L12 15l-4 1 1-4 9.5-9.5z"></path>
                                            </svg>
                                        </a>
                                        <a href="?delete=<?php echo $train['id']; ?>" 
                                           class="btn btn-danger btn-sm btn-icon"
                                           data-confirm="Are you sure you want to delete this train?"
                                           title="Delete">
                                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                                <polyline points="3 6 5 6 21 6"></polyline>
                                                <path d="M19 6v14a2 2 0 0 1-2 2H7a2 2 0 0 1-2-2V6m3 0V4a2 2 0 0 1 2-2h4a2 2 0 0 1 2 2v2"></path>
                                            </svg>
                                        </a>
                                    </div>
                                </td>
                            </tr>
                        <?php endwhile; ?>
                    <?php else: ?>
                        <tr>
                            <td colspan="6" style="text-align: center; padding: 60px 20px;">
                                <div class="empty-state">
                                    <svg width="80" height="80" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                        <rect x="4" y="4" width="16" height="16" rx="2" ry="2"></rect>
                                    </svg>
                                    <h3>No trains found</h3>
                                    <p>Click "Add New Train" to get started</p>
                                </div>
                            </td>
                        </tr>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>

<!-- Add/Edit Modal -->
<div class="admin-modal" id="trainModal">
    <div class="modal-dialog">
        <div class="modal-header">
            <h3 id="modalTitle">Add New Train</h3>
            <button class="modal-close" onclick="closeModal()">&times;</button>
        </div>
        <form method="POST">
            <div class="modal-body">
                <input type="hidden" name="train_id" id="train_id">
                
                <div class="form-group-admin">
                    <label>Train Number *</label>
                    <input type="text" name="train_number" id="train_number" required placeholder="e.g., KA-006">
                </div>
                
                <div class="form-group-admin">
                    <label>Train Name *</label>
                    <input type="text" name="train_name" id="train_name" required placeholder="e.g., Argo Bromo">
                </div>
                
                <div class="form-group-admin">
                    <label>Train Type *</label>
                    <select name="train_type" id="train_type" required>
                        <option value="">Select Type</option>
                        <option value="Express">Express</option>
                        <option value="Business">Business</option>
                        <option value="Economy">Economy</option>
                    </select>
                </div>
                
                <div class="form-group-admin">
                    <label>Capacity *</label>
                    <input type="number" name="capacity" id="capacity" required min="1" placeholder="e.g., 300">
                    <small class="form-hint">Total number of seats available</small>
                </div>
            </div>
            
            <div class="modal-footer">
                <button type="button" class="btn btn-outline" onclick="closeModal()">Cancel</button>
                <button type="submit" name="save_train" class="btn btn-primary">
                    <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <polyline points="20 6 9 17 4 12"></polyline>
                    </svg>
                    Save Train
                </button>
            </div>
        </form>
    </div>
</div>

<script>
function openModal() {
    document.getElementById('trainModal').classList.add('active');
    document.getElementById('modalTitle').textContent = 'Add New Train';
    document.getElementById('train_id').value = '';
    document.getElementById('train_number').value = '';
    document.getElementById('train_name').value = '';
    document.getElementById('train_type').value = '';
    document.getElementById('capacity').value = '';
}

function closeModal() {
    document.getElementById('trainModal').classList.remove('active');
}

function editTrain(train) {
    document.getElementById('trainModal').classList.add('active');
    document.getElementById('modalTitle').textContent = 'Edit Train';
    document.getElementById('train_id').value = train.id;
    document.getElementById('train_number').value = train.train_number;
    document.getElementById('train_name').value = train.train_name;
    document.getElementById('train_type').value = train.train_type;
    document.getElementById('capacity').value = train.capacity;
}

// Close modal on outside click
document.getElementById('trainModal').addEventListener('click', function(e) {
    if(e.target === this) {
        closeModal();
    }
});
</script>

<?php include 'includes/admin-footer.php'; ?>