<?php
$page_title = "Manage Stations";
include 'includes/admin-header.php';

// Handle Add/Edit
if(isset($_POST['save_station'])) {
    $station_id = isset($_POST['station_id']) ? (int)$_POST['station_id'] : 0;
    $station_name = mysqli_real_escape_string($conn, $_POST['station_name']);
    $city = mysqli_real_escape_string($conn, $_POST['city']);
    $code = mysqli_real_escape_string($conn, $_POST['code']);
    $status = mysqli_real_escape_string($conn, $_POST['status']);
    
    if($station_id > 0) {
        // Update
        $query = "UPDATE stations SET station_name = ?, city = ?, code = ?, status = ? WHERE id = ?";
        $stmt = mysqli_prepare($conn, $query);
        mysqli_stmt_bind_param($stmt, "ssssi", $station_name, $city, $code, $status, $station_id);
        $message = "Station updated successfully!";
    } else {
        // Insert
        $query = "INSERT INTO stations (station_name, city, code, status) VALUES (?, ?, ?, ?)";
        $stmt = mysqli_prepare($conn, $query);
        mysqli_stmt_bind_param($stmt, "ssss", $station_name, $city, $code, $status);
        $message = "Station added successfully!";
    }
    
    if(mysqli_stmt_execute($stmt)) {
        $_SESSION['success'] = $message;
    } else {
        $_SESSION['error'] = "Error: " . mysqli_error($conn);
    }
    
    header("Location: stations.php");
    exit();
}

// Handle Delete
if(isset($_GET['delete'])) {
    $station_id = (int)$_GET['delete'];
    
    $delete_query = "DELETE FROM stations WHERE id = ?";
    $stmt = mysqli_prepare($conn, $delete_query);
    mysqli_stmt_bind_param($stmt, "i", $station_id);
    
    if(mysqli_stmt_execute($stmt)) {
        $_SESSION['success'] = "Station deleted successfully!";
    } else {
        $_SESSION['error'] = "Cannot delete station. It may be associated with schedules.";
    }
    
    header("Location: stations.php");
    exit();
}

// Get all stations
$query = "SELECT * FROM stations ORDER BY station_name ASC";
$stations = mysqli_query($conn, $query);

include 'includes/admin-sidebar.php';
?>

<div class="admin-container">
    <div class="page-header">
        <div>
            <h1>Manage Stations</h1>
            <p>Add, edit, or remove railway stations</p>
        </div>
        <button class="btn btn-primary" onclick="openModal()">
            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <line x1="12" y1="5" x2="12" y2="19"></line>
                <line x1="5" y1="12" x2="19" y2="12"></line>
            </svg>
            Add New Station
        </button>
    </div>

    <?php if(isset($_SESSION['success'])): ?>
        <div class="alert-success">
            <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <polyline points="20 6 9 17 4 12"></polyline>
            </svg>
            <?php echo $_SESSION['success']; unset($_SESSION['success']); ?>
        </div>
    <?php endif; ?>

    <?php if(isset($_SESSION['error'])): ?>
        <div class="alert-error">
            <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <circle cx="12" cy="12" r="10"></circle>
                <line x1="15" y1="9" x2="9" y2="15"></line>
                <line x1="9" y1="9" x2="15" y2="15"></line>
            </svg>
            <?php echo $_SESSION['error']; unset($_SESSION['error']); ?>
        </div>
    <?php endif; ?>

    <!-- Stations Table -->
    <div class="admin-card">
        <div class="table-container">
            <table class="data-table">
                <thead>
                    <tr>
                        <th>Code</th>
                        <th>Station Name</th>
                        <th>City</th>
                        <th>Status</th>
                        <th>Created At</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if(mysqli_num_rows($stations) > 0): ?>
                        <?php while($station = mysqli_fetch_assoc($stations)): ?>
                            <tr>
                                <td><strong><?php echo htmlspecialchars($station['code']); ?></strong></td>
                                <td><?php echo htmlspecialchars($station['station_name']); ?></td>
                                <td><?php echo htmlspecialchars($station['city']); ?></td>
                                <td>
                                    <span class="status-badge status-<?php echo strtolower($station['status']); ?>">
                                        <?php echo $station['status']; ?>
                                    </span>
                                </td>
                                <td><?php echo date('d M Y', strtotime($station['created_at'])); ?></td>
                                <td>
                                    <div class="table-actions">
                                        <a href="#" 
                                           class="btn btn-primary btn-sm btn-icon"
                                           onclick="editStation(<?php echo htmlspecialchars(json_encode($station)); ?>); return false;"
                                           title="Edit">
                                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                                <path d="M11 4H4a2 2 0 0 0-2 2v14a2 2 0 0 0 2 2h14a2 2 0 0 0 2-2v-7"></path>
                                                <path d="M18.5 2.5a2.121 2.121 0 0 1 3 3L12 15l-4 1 1-4 9.5-9.5z"></path>
                                            </svg>
                                        </a>
                                        <a href="?delete=<?php echo $station['id']; ?>" 
                                           class="btn btn-danger btn-sm btn-icon"
                                           data-confirm="Are you sure you want to delete this station?"
                                           title="Delete">
                                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                                <polyline points="3 6 5 6 21 6"></polyline>
                                                <path d="M19 6v14a2 2 0 0 1-2 2H7a2 2 0 0 1-2-2V6m3 0V4a2 2 0 0 1 2-2h4a2 2 0 0 1 2 2v2"></path>
                                            </svg>
                                        </a>
                                    </div>
                                </td>
                            </tr>
                        <?php endwhile; ?>
                    <?php else: ?>
                        <tr>
                            <td colspan="6" style="text-align: center; padding: 60px 20px;">
                                <div class="empty-state">
                                    <svg width="80" height="80" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                        <path d="M21 10c0 7-9 13-9 13s-9-6-9-13a9 9 0 0 1 18 0z"></path>
                                        <circle cx="12" cy="10" r="3"></circle>
                                    </svg>
                                    <h3>No stations found</h3>
                                    <p>Click "Add New Station" to get started</p>
                                </div>
                            </td>
                        </tr>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>

<!-- Add/Edit Modal -->
<div class="admin-modal" id="stationModal">
    <div class="modal-dialog">
        <div class="modal-header">
            <h3 id="modalTitle">Add New Station</h3>
            <button class="modal-close" onclick="closeModal()">&times;</button>
        </div>
        <form method="POST">
            <div class="modal-body">
                <input type="hidden" name="station_id" id="station_id">
                
                <div class="form-group-admin">
                    <label>Station Name *</label>
                    <input type="text" name="station_name" id="station_name" required placeholder="e.g., Gambir Station">
                </div>
                
                <div class="form-group-admin">
                    <label>City *</label>
                    <input type="text" name="city" id="city" required placeholder="e.g., Jakarta">
                </div>
                
                <div class="form-group-admin">
                    <label>Station Code *</label>
                    <input type="text" name="code" id="code" required placeholder="e.g., GMR" maxlength="10">
                    <small class="form-hint">3-letter station code</small>
                </div>
                
                <div class="form-group-admin">
                    <label>Status *</label>
                    <select name="status" id="status" required>
                        <option value="Active">Active</option>
                        <option value="Deactive">Deactive</option>
                    </select>
                </div>
            </div>
            
            <div class="modal-footer">
                <button type="button" class="btn btn-outline" onclick="closeModal()">Cancel</button>
                <button type="submit" name="save_station" class="btn btn-primary">
                    <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <polyline points="20 6 9 17 4 12"></polyline>
                    </svg>
                    Save Station
                </button>
            </div>
        </form>
    </div>
</div>

<script>
function openModal() {
    document.getElementById('stationModal').classList.add('active');
    document.getElementById('modalTitle').textContent = 'Add New Station';
    document.getElementById('station_id').value = '';
    document.getElementById('station_name').value = '';
    document.getElementById('city').value = '';
    document.getElementById('code').value = '';
    document.getElementById('status').value = 'Active';
}

function closeModal() {
    document.getElementById('stationModal').classList.remove('active');
}

function editStation(station) {
    document.getElementById('stationModal').classList.add('active');
    document.getElementById('modalTitle').textContent = 'Edit Station';
    document.getElementById('station_id').value = station.id;
    document.getElementById('station_name').value = station.station_name;
    document.getElementById('city').value = station.city;
    document.getElementById('code').value = station.code;
    document.getElementById('status').value = station.status;
}

// Close modal on outside click
document.getElementById('stationModal').addEventListener('click', function(e) {
    if(e.target === this) {
        closeModal();
    }
});
</script>

<?php include 'includes/admin-footer.php'; ?>
                