<?php
$page_title = "Manage Schedules";
include 'includes/admin-header.php';

// Handle Add/Edit
if(isset($_POST['save_schedule'])) {
    $schedule_id = isset($_POST['schedule_id']) ? (int)$_POST['schedule_id'] : 0;
    $train_id = (int)$_POST['train_id'];
    $origin_station_id = (int)$_POST['origin_station_id'];
    $destination_station_id = (int)$_POST['destination_station_id'];
    $departure_time = mysqli_real_escape_string($conn, $_POST['departure_time']);
    $arrival_time = mysqli_real_escape_string($conn, $_POST['arrival_time']);
    $duration = mysqli_real_escape_string($conn, $_POST['duration']);
    $price = (float)$_POST['price'];
    $available_seats = (int)$_POST['available_seats'];
    $status = mysqli_real_escape_string($conn, $_POST['status']);
    
    if($schedule_id > 0) {
        // Update
        $query = "UPDATE schedules SET train_id = ?, origin_station_id = ?, destination_station_id = ?, 
                  departure_time = ?, arrival_time = ?, duration = ?, price = ?, available_seats = ?, status = ? 
                  WHERE id = ?";
        $stmt = mysqli_prepare($conn, $query);
        mysqli_stmt_bind_param($stmt, "iiisssdiis", $train_id, $origin_station_id, $destination_station_id, 
                               $departure_time, $arrival_time, $duration, $price, $available_seats, $status, $schedule_id);
        $message = "Schedule updated successfully!";
    } else {
        // Insert
        $query = "INSERT INTO schedules (train_id, origin_station_id, destination_station_id, departure_time, 
                  arrival_time, duration, price, available_seats, status) 
                  VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)";
        $stmt = mysqli_prepare($conn, $query);
        mysqli_stmt_bind_param($stmt, "iiisssdis", $train_id, $origin_station_id, $destination_station_id, 
                               $departure_time, $arrival_time, $duration, $price, $available_seats, $status);
        $message = "Schedule added successfully!";
    }
    
    if(mysqli_stmt_execute($stmt)) {
        $_SESSION['success'] = $message;
    } else {
        $_SESSION['error'] = "Error: " . mysqli_error($conn);
    }
    
    header("Location: schedules.php");
    exit();
}

// Handle Delete
if(isset($_GET['delete'])) {
    $schedule_id = (int)$_GET['delete'];
    
    $delete_query = "DELETE FROM schedules WHERE id = ?";
    $stmt = mysqli_prepare($conn, $delete_query);
    mysqli_stmt_bind_param($stmt, "i", $schedule_id);
    
    if(mysqli_stmt_execute($stmt)) {
        $_SESSION['success'] = "Schedule deleted successfully!";
    } else {
        $_SESSION['error'] = "Cannot delete schedule. It may have associated bookings.";
    }
    
    header("Location: schedules.php");
    exit();
}

// Get all schedules with related data
$query = "SELECT s.*, t.train_name, t.train_number,
          s1.station_name as origin_name, s1.code as origin_code,
          s2.station_name as destination_name, s2.code as destination_code
          FROM schedules s
          JOIN trains t ON s.train_id = t.id
          JOIN stations s1 ON s.origin_station_id = s1.id
          JOIN stations s2 ON s.destination_station_id = s2.id
          ORDER BY s.departure_time ASC";
$schedules = mysqli_query($conn, $query);

// Get trains for dropdown
$trains_query = "SELECT id, train_number, train_name FROM trains ORDER BY train_number ASC";
$trains = mysqli_query($conn, $trains_query);

// Get stations for dropdown
$stations_query = "SELECT id, station_name, code FROM stations WHERE status = 'Active' ORDER BY station_name ASC";
$stations = mysqli_query($conn, $stations_query);

include 'includes/admin-sidebar.php';
?>

<div class="admin-container">
    <div class="page-header">
        <div>
            <h1>Manage Schedules</h1>
            <p>Add, edit, or remove train schedules</p>
        </div>
        <button class="btn btn-primary" onclick="openModal()">
            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <line x1="12" y1="5" x2="12" y2="19"></line>
                <line x1="5" y1="12" x2="19" y2="12"></line>
            </svg>
            Add New Schedule
        </button>
    </div>

    <?php if(isset($_SESSION['success'])): ?>
        <div class="alert-success">
            <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <polyline points="20 6 9 17 4 12"></polyline>
            </svg>
            <?php echo $_SESSION['success']; unset($_SESSION['success']); ?>
        </div>
    <?php endif; ?>

    <?php if(isset($_SESSION['error'])): ?>
        <div class="alert-error">
            <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <circle cx="12" cy="12" r="10"></circle>
                <line x1="15" y1="9" x2="9" y2="15"></line>
                <line x1="9" y1="9" x2="15" y2="15"></line>
            </svg>
            <?php echo $_SESSION['error']; unset($_SESSION['error']); ?>
        </div>
    <?php endif; ?>

    <!-- Schedules Table -->
    <div class="admin-card">
        <div class="table-container">
            <table class="data-table">
                <thead>
                    <tr>
                        <th>Train</th>
                        <th>Route</th>
                        <th>Departure</th>
                        <th>Arrival</th>
                        <th>Duration</th>
                        <th>Price</th>
                        <th>Available Seats</th>
                        <th>Status</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if(mysqli_num_rows($schedules) > 0): ?>
                        <?php while($schedule = mysqli_fetch_assoc($schedules)): ?>
                            <tr>
                                <td>
                                    <strong><?php echo htmlspecialchars($schedule['train_number']); ?></strong><br>
                                    <small style="color: var(--gray);"><?php echo htmlspecialchars($schedule['train_name']); ?></small>
                                </td>
                                <td>
                                    <strong><?php echo htmlspecialchars($schedule['origin_code']); ?></strong> → 
                                    <strong><?php echo htmlspecialchars($schedule['destination_code']); ?></strong><br>
                                    <small style="color: var(--gray);">
                                        <?php echo htmlspecialchars($schedule['origin_name']); ?> - 
                                        <?php echo htmlspecialchars($schedule['destination_name']); ?>
                                    </small>
                                </td>
                                <td><?php echo date('H:i', strtotime($schedule['departure_time'])); ?></td>
                                <td><?php echo date('H:i', strtotime($schedule['arrival_time'])); ?></td>
                                <td><?php echo htmlspecialchars($schedule['duration']); ?></td>
                                <td><strong>Rp <?php echo number_format($schedule['price'], 0, ',', '.'); ?></strong></td>
                                <td><?php echo $schedule['available_seats']; ?> seats</td>
                                <td>
                                    <span class="status-badge status-<?php echo $schedule['status']; ?>">
                                        <?php echo ucfirst($schedule['status']); ?>
                                    </span>
                                </td>
                                <td>
                                    <div class="table-actions">
                                        <a href="#" 
                                           class="btn btn-primary btn-sm btn-icon"
                                           onclick="editSchedule(<?php echo htmlspecialchars(json_encode($schedule)); ?>); return false;"
                                           title="Edit">
                                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                                <path d="M11 4H4a2 2 0 0 0-2 2v14a2 2 0 0 0 2 2h14a2 2 0 0 0 2-2v-7"></path>
                                                <path d="M18.5 2.5a2.121 2.121 0 0 1 3 3L12 15l-4 1 1-4 9.5-9.5z"></path>
                                            </svg>
                                        </a>
                                        <a href="?delete=<?php echo $schedule['id']; ?>" 
                                           class="btn btn-danger btn-sm btn-icon"
                                           data-confirm="Are you sure you want to delete this schedule?"
                                           title="Delete">
                                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                                <polyline points="3 6 5 6 21 6"></polyline>
                                                <path d="M19 6v14a2 2 0 0 1-2 2H7a2 2 0 0 1-2-2V6m3 0V4a2 2 0 0 1 2-2h4a2 2 0 0 1 2 2v2"></path>
                                            </svg>
                                        </a>
                                    </div>
                                </td>
                            </tr>
                        <?php endwhile; ?>
                    <?php else: ?>
                        <tr>
                            <td colspan="9" style="text-align: center; padding: 60px 20px;">
                                <div class="empty-state">
                                    <svg width="80" height="80" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                        <rect x="3" y="4" width="18" height="18" rx="2" ry="2"></rect>
                                        <line x1="16" y1="2" x2="16" y2="6"></line>
                                        <line x1="8" y1="2" x2="8" y2="6"></line>
                                        <line x1="3" y1="10" x2="21" y2="10"></line>
                                    </svg>
                                    <h3>No schedules found</h3>
                                    <p>Click "Add New Schedule" to get started</p>
                                </div>
                            </td>
                        </tr>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>

<!-- Add/Edit Modal -->
<div class="admin-modal" id="scheduleModal">
    <div class="modal-dialog" style="max-width: 700px;">
        <div class="modal-header">
            <h3 id="modalTitle">Add New Schedule</h3>
            <button class="modal-close" onclick="closeModal()">&times;</button>
        </div>
        <form method="POST">
            <div class="modal-body">
                <input type="hidden" name="schedule_id" id="schedule_id">
                
                <div class="form-row">
                    <div class="form-group-admin">
                        <label>Train *</label>
                        <select name="train_id" id="train_id" required>
                            <option value="">Select Train</option>
                            <?php mysqli_data_seek($trains, 0); while($train = mysqli_fetch_assoc($trains)): ?>
                                <option value="<?php echo $train['id']; ?>">
                                    <?php echo htmlspecialchars($train['train_number'] . ' - ' . $train['train_name']); ?>
                                </option>
                            <?php endwhile; ?>
                        </select>
                    </div>
                    
                    <div class="form-group-admin">
                        <label>Status *</label>
                        <select name="status" id="status" required>
                            <option value="active">Active</option>
                            <option value="inactive">Inactive</option>
                        </select>
                    </div>
                </div>
                
                <div class="form-row">
                    <div class="form-group-admin">
                        <label>Origin Station *</label>
                        <select name="origin_station_id" id="origin_station_id" required>
                            <option value="">Select Origin</option>
                            <?php mysqli_data_seek($stations, 0); while($station = mysqli_fetch_assoc($stations)): ?>
                                <option value="<?php echo $station['id']; ?>">
                                    <?php echo htmlspecialchars($station['code'] . ' - ' . $station['station_name']); ?>
                                </option>
                            <?php endwhile; ?>
                        </select>
                    </div>
                    
                    <div class="form-group-admin">
                        <label>Destination Station *</label>
                        <select name="destination_station_id" id="destination_station_id" required>
                            <option value="">Select Destination</option>
                            <?php mysqli_data_seek($stations, 0); while($station = mysqli_fetch_assoc($stations)): ?>
                                <option value="<?php echo $station['id']; ?>">
                                    <?php echo htmlspecialchars($station['code'] . ' - ' . $station['station_name']); ?>
                                </option>
                            <?php endwhile; ?>
                        </select>
                    </div>
                </div>
                
                <div class="form-row">
                    <div class="form-group-admin">
                        <label>Departure Time *</label>
                        <input type="time" name="departure_time" id="departure_time" required>
                    </div>
                    
                    <div class="form-group-admin">
                        <label>Arrival Time *</label>
                        <input type="time" name="arrival_time" id="arrival_time" required>
                    </div>
                </div>
                
                <div class="form-row">
                    <div class="form-group-admin">
                        <label>Duration *</label>
                        <input type="text" name="duration" id="duration" required placeholder="e.g., 3h 30m">
                        <small class="form-hint">Format: 3h 30m</small>
                    </div>
                    
                    <div class="form-group-admin">
                        <label>Price (Rp) *</label>
                        <input type="number" name="price" id="price" required min="0" step="1000" placeholder="150000">
                    </div>
                </div>
                
                <div class="form-group-admin">
                    <label>Available Seats *</label>
                    <input type="number" name="available_seats" id="available_seats" required min="1" placeholder="300">
                </div>
            </div>
            
            <div class="modal-footer">
                <button type="button" class="btn btn-outline" onclick="closeModal()">Cancel</button>
                <button type="submit" name="save_schedule" class="btn btn-primary">
                    <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <polyline points="20 6 9 17 4 12"></polyline>
                    </svg>
                    Save Schedule
                </button>
            </div>
        </form>
    </div>
</div>

<script>
function openModal() {
    document.getElementById('scheduleModal').classList.add('active');
    document.getElementById('modalTitle').textContent = 'Add New Schedule';
    document.getElementById('schedule_id').value = '';
    document.getElementById('train_id').value = '';
    document.getElementById('origin_station_id').value = '';
    document.getElementById('destination_station_id').value = '';
    document.getElementById('departure_time').value = '';
    document.getElementById('arrival_time').value = '';
    document.getElementById('duration').value = '';
    document.getElementById('price').value = '';
    document.getElementById('available_seats').value = '';
    document.getElementById('status').value = 'active';
}

function closeModal() {
    document.getElementById('scheduleModal').classList.remove('active');
}

function editSchedule(schedule) {
    document.getElementById('scheduleModal').classList.add('active');
    document.getElementById('modalTitle').textContent = 'Edit Schedule';
    document.getElementById('schedule_id').value = schedule.id;
    document.getElementById('train_id').value = schedule.train_id;
    document.getElementById('origin_station_id').value = schedule.origin_station_id;
    document.getElementById('destination_station_id').value = schedule.destination_station_id;
    document.getElementById('departure_time').value = schedule.departure_time;
    document.getElementById('arrival_time').value = schedule.arrival_time;
    document.getElementById('duration').value = schedule.duration;
    document.getElementById('price').value = schedule.price;
    document.getElementById('available_seats').value = schedule.available_seats;
    document.getElementById('status').value = schedule.status;
}

// Close modal on outside click
document.getElementById('scheduleModal').addEventListener('click', function(e) {
    if(e.target === this) {
        closeModal();
    }
});
</script>

<?php include 'includes/admin-footer.php'; ?>