<?php
$page_title = "Manage Promotions";
include 'includes/admin-header.php';

// Handle Add/Edit
if(isset($_POST['save_promotion'])) {
    $promo_id = isset($_POST['promo_id']) ? (int)$_POST['promo_id'] : 0;
    $title = mysqli_real_escape_string($conn, $_POST['title']);
    $description = mysqli_real_escape_string($conn, $_POST['description']);
    $promo_type = mysqli_real_escape_string($conn, $_POST['promo_type']);
    $discount_percentage = (int)$_POST['discount_percentage'];
    $start_date = mysqli_real_escape_string($conn, $_POST['start_date']);
    $end_date = mysqli_real_escape_string($conn, $_POST['end_date']);
    $status = mysqli_real_escape_string($conn, $_POST['status']);
    
    // Handle image upload
    $image = '';
    if(isset($_FILES['image']) && $_FILES['image']['error'] == 0) {
        $allowed = ['jpg', 'jpeg', 'png', 'gif'];
        $filename = $_FILES['image']['name'];
        $filetype = pathinfo($filename, PATHINFO_EXTENSION);
        
        if(in_array(strtolower($filetype), $allowed)) {
            $newname = 'promo_' . time() . '.' . $filetype;
            $upload_path = '../assets/img/promotions/';
            
            // Create directory if not exists
            if(!is_dir($upload_path)) {
                mkdir($upload_path, 0777, true);
            }
            
            if(move_uploaded_file($_FILES['image']['tmp_name'], $upload_path . $newname)) {
                $image = $newname;
                
                // Delete old image if updating
                if($promo_id > 0) {
                    $old_query = "SELECT image FROM promotions WHERE id = ?";
                    $old_stmt = mysqli_prepare($conn, $old_query);
                    mysqli_stmt_bind_param($old_stmt, "i", $promo_id);
                    mysqli_stmt_execute($old_stmt);
                    $old_result = mysqli_stmt_get_result($old_stmt);
                    $old_data = mysqli_fetch_assoc($old_result);
                    
                    if($old_data['image'] && file_exists($upload_path . $old_data['image'])) {
                        unlink($upload_path . $old_data['image']);
                    }
                }
            }
        }
    }
    
    if($promo_id > 0) {
        // Update
        if($image) {
            $query = "UPDATE promotions SET title = ?, description = ?, promo_type = ?, discount_percentage = ?, 
                      start_date = ?, end_date = ?, status = ?, image = ? WHERE id = ?";
            $stmt = mysqli_prepare($conn, $query);
            mysqli_stmt_bind_param($stmt, "sssissssi", $title, $description, $promo_type, $discount_percentage, 
                                   $start_date, $end_date, $status, $image, $promo_id);
        } else {
            $query = "UPDATE promotions SET title = ?, description = ?, promo_type = ?, discount_percentage = ?, 
                      start_date = ?, end_date = ?, status = ? WHERE id = ?";
            $stmt = mysqli_prepare($conn, $query);
            mysqli_stmt_bind_param($stmt, "sssisssi", $title, $description, $promo_type, $discount_percentage, 
                                   $start_date, $end_date, $status, $promo_id);
        }
        $message = "Promotion updated successfully!";
    } else {
        // Insert
        $query = "INSERT INTO promotions (title, description, promo_type, discount_percentage, start_date, end_date, status, image) 
                  VALUES (?, ?, ?, ?, ?, ?, ?, ?)";
        $stmt = mysqli_prepare($conn, $query);
        mysqli_stmt_bind_param($stmt, "ssisssss", $title, $description, $promo_type, $discount_percentage, 
                               $start_date, $end_date, $status, $image);
        $message = "Promotion added successfully!";
    }
    
    if(mysqli_stmt_execute($stmt)) {
        $_SESSION['success'] = $message;
    } else {
        $_SESSION['error'] = "Error: " . mysqli_error($conn);
    }
    
    header("Location: promotions.php");
    exit();
}

// Handle Delete
if(isset($_GET['delete'])) {
    $promo_id = (int)$_GET['delete'];
    
    // Get image filename before delete
    $img_query = "SELECT image FROM promotions WHERE id = ?";
    $img_stmt = mysqli_prepare($conn, $img_query);
    mysqli_stmt_bind_param($img_stmt, "i", $promo_id);
    mysqli_stmt_execute($img_stmt);
    $img_result = mysqli_stmt_get_result($img_stmt);
    $img_data = mysqli_fetch_assoc($img_result);
    
    $delete_query = "DELETE FROM promotions WHERE id = ?";
    $stmt = mysqli_prepare($conn, $delete_query);
    mysqli_stmt_bind_param($stmt, "i", $promo_id);
    
    if(mysqli_stmt_execute($stmt)) {
        // Delete image file
        if($img_data['image']) {
            $img_path = '../assets/img/promotions/' . $img_data['image'];
            if(file_exists($img_path)) {
                unlink($img_path);
            }
        }
        $_SESSION['success'] = "Promotion deleted successfully!";
    } else {
        $_SESSION['error'] = "Failed to delete promotion.";
    }
    
    header("Location: promotions.php");
    exit();
}

// Get all promotions
$query = "SELECT * FROM promotions ORDER BY created_at DESC";
$promotions = mysqli_query($conn, $query);

include 'includes/admin-sidebar.php';
?>

<div class="admin-container">
    <div class="page-header">
        <div>
            <h1>Manage Promotions</h1>
            <p>Add, edit, or remove promotional offers</p>
        </div>
        <button class="btn btn-primary" onclick="openModal()">
            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <line x1="12" y1="5" x2="12" y2="19"></line>
                <line x1="5" y1="12" x2="19" y2="12"></line>
            </svg>
            Add New Promotion
        </button>
    </div>

    <?php if(isset($_SESSION['success'])): ?>
        <div class="alert-success">
            <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <polyline points="20 6 9 17 4 12"></polyline>
            </svg>
            <?php echo $_SESSION['success']; unset($_SESSION['success']); ?>
        </div>
    <?php endif; ?>

    <?php if(isset($_SESSION['error'])): ?>
        <div class="alert-error">
            <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <circle cx="12" cy="12" r="10"></circle>
                <line x1="15" y1="9" x2="9" y2="15"></line>
                <line x1="9" y1="9" x2="15" y2="15"></line>
            </svg>
            <?php echo $_SESSION['error']; unset($_SESSION['error']); ?>
        </div>
    <?php endif; ?>

    <!-- Promotions Table -->
    <div class="admin-card">
        <div class="table-container">
            <table class="data-table">
                <thead>
                    <tr>
                        <th>Image</th>
                        <th>Title</th>
                        <th>Type</th>
                        <th>Discount</th>
                        <th>Valid Period</th>
                        <th>Status</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if(mysqli_num_rows($promotions) > 0): ?>
                        <?php while($promo = mysqli_fetch_assoc($promotions)): ?>
                            <tr>
                                <td>
                                    <?php if($promo['image']): ?>
                                        <img src="<?php echo BASE_URL; ?>assets/img/promotions/<?php echo $promo['image']; ?>" 
                                             alt="Promo" 
                                             style="width: 60px; height: 60px; object-fit: cover; border-radius: 8px;">
                                    <?php else: ?>
                                        <div style="width: 60px; height: 60px; background: var(--light-gray); border-radius: 8px; display: flex; align-items: center; justify-content: center; color: var(--gray);">
                                            No Image
                                        </div>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <strong><?php echo htmlspecialchars($promo['title']); ?></strong><br>
                                    <small style="color: var(--gray);">
                                        <?php echo htmlspecialchars(substr($promo['description'], 0, 50)); ?>...
                                    </small>
                                </td>
                                <td>
                                    <span class="status-badge status-active">
                                        <?php echo $promo['promo_type']; ?>
                                    </span>
                                </td>
                                <td><strong><?php echo $promo['discount_percentage']; ?>% OFF</strong></td>
                                <td>
                                    <?php echo date('d M Y', strtotime($promo['start_date'])); ?><br>
                                    <small style="color: var(--gray);">to <?php echo date('d M Y', strtotime($promo['end_date'])); ?></small>
                                </td>
                                <td>
                                    <span class="status-badge status-<?php echo $promo['status']; ?>">
                                        <?php echo ucfirst($promo['status']); ?>
                                    </span>
                                </td>
                                <td>
                                    <div class="table-actions">
                                        <a href="#" 
                                           class="btn btn-primary btn-sm btn-icon"
                                           onclick="editPromotion(<?php echo htmlspecialchars(json_encode($promo)); ?>); return false;"
                                           title="Edit">
                                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                                <path d="M11 4H4a2 2 0 0 0-2 2v14a2 2 0 0 0 2 2h14a2 2 0 0 0 2-2v-7"></path>
                                                <path d="M18.5 2.5a2.121 2.121 0 0 1 3 3L12 15l-4 1 1-4 9.5-9.5z"></path>
                                            </svg>
                                        </a>
                                        <a href="?delete=<?php echo $promo['id']; ?>" 
                                           class="btn btn-danger btn-sm btn-icon"
                                           data-confirm="Are you sure you want to delete this promotion?"
                                           title="Delete">
                                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                                <polyline points="3 6 5 6 21 6"></polyline>
                                                <path d="M19 6v14a2 2 0 0 1-2 2H7a2 2 0 0 1-2-2V6m3 0V4a2 2 0 0 1 2-2h4a2 2 0 0 1 2 2v2"></path>
                                            </svg>
                                        </a>
                                    </div>
                                </td>
                            </tr>
                        <?php endwhile; ?>
                    <?php else: ?>
                        <tr>
                            <td colspan="7" style="text-align: center; padding: 60px 20px;">
                                <div class="empty-state">
                                    <svg width="80" height="80" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                        <path d="M20.59 13.41l-7.17 7.17a2 2 0 0 1-2.83 0L2 12V2h10l8.59 8.59a2 2 0 0 1 0 2.82z"></path>
                                        <line x1="7" y1="7" x2="7.01" y2="7"></line>
                                    </svg>
                                    <h3>No promotions found</h3>
                                    <p>Click "Add New Promotion" to get started</p>
                                </div>
                            </td>
                        </tr>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>

<!-- Add/Edit Modal -->
<div class="admin-modal" id="promoModal">
    <div class="modal-dialog" style="max-width: 700px;">
        <div class="modal-header">
            <h3 id="modalTitle">Add New Promotion</h3>
            <button class="modal-close" onclick="closeModal()">&times;</button>
        </div>
        <form method="POST" enctype="multipart/form-data">
            <div class="modal-body">
                <input type="hidden" name="promo_id" id="promo_id">
                
                <div class="form-group-admin">
                    <label>Promotion Image</label>
                    <input type="file" name="image" id="image" accept="image/*" onchange="previewImage(this)">
                    <small class="form-hint">Recommended size: 800x400px (JPG, PNG, GIF - Max 2MB)</small>
                    <div id="imagePreview" style="margin-top: 12px;"></div>
                </div>
                
                <div class="form-group-admin">
                    <label>Promotion Title *</label>
                    <input type="text" name="title" id="title" required placeholder="e.g., Weekend Special - 25% Off">
                </div>
                
                <div class="form-group-admin">
                    <label>Description *</label>
                    <textarea name="description" id="description" required placeholder="Describe the promotion details..."></textarea>
                </div>
                
                <div class="form-row">
                    <div class="form-group-admin">
                        <label>Promotion Type *</label>
                        <select name="promo_type" id="promo_type" required>
                            <option value="">Select Type</option>
                            <option value="All">All Passengers</option>
                            <option value="Seasonal">Seasonal</option>
                            <option value="Student">Student</option>
                            <option value="Group">Group</option>
                        </select>
                    </div>
                    
                    <div class="form-group-admin">
                        <label>Discount Percentage *</label>
                        <input type="number" name="discount_percentage" id="discount_percentage" required min="1" max="100" placeholder="25">
                        <small class="form-hint">Enter value between 1-100</small>
                    </div>
                </div>
                
                <div class="form-row">
                    <div class="form-group-admin">
                        <label>Start Date *</label>
                        <input type="date" name="start_date" id="start_date" required>
                    </div>
                    
                    <div class="form-group-admin">
                        <label>End Date *</label>
                        <input type="date" name="end_date" id="end_date" required>
                    </div>
                </div>
                
                <div class="form-group-admin">
                    <label>Status *</label>
                    <select name="status" id="status" required>
                        <option value="active">Active</option>
                        <option value="inactive">Inactive</option>
                    </select>
                </div>
            </div>
            
            <div class="modal-footer">
                <button type="button" class="btn btn-outline" onclick="closeModal()">Cancel</button>
                <button type="submit" name="save_promotion" class="btn btn-primary">
                    <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <polyline points="20 6 9 17 4 12"></polyline>
                    </svg>
                    Save Promotion
                </button>
            </div>
        </form>
    </div>
</div>

<script>
function openModal() {
    document.getElementById('promoModal').classList.add('active');
    document.getElementById('modalTitle').textContent = 'Add New Promotion';
    document.getElementById('promo_id').value = '';
    document.getElementById('title').value = '';
    document.getElementById('description').value = '';
    document.getElementById('promo_type').value = '';
    document.getElementById('discount_percentage').value = '';
    document.getElementById('start_date').value = '';
    document.getElementById('end_date').value = '';
    document.getElementById('status').value = 'active';
    document.getElementById('image').value = '';
    document.getElementById('imagePreview').innerHTML = '';
}

function closeModal() {
    document.getElementById('promoModal').classList.remove('active');
}

function editPromotion(promo) {
    document.getElementById('promoModal').classList.add('active');
    document.getElementById('modalTitle').textContent = 'Edit Promotion';
    document.getElementById('promo_id').value = promo.id;
    document.getElementById('title').value = promo.title;
    document.getElementById('description').value = promo.description;
    document.getElementById('promo_type').value = promo.promo_type;
    document.getElementById('discount_percentage').value = promo.discount_percentage;
    document.getElementById('start_date').value = promo.start_date;
    document.getElementById('end_date').value = promo.end_date;
    document.getElementById('status').value = promo.status;
    
    // Show existing image
    if(promo.image) {
        document.getElementById('imagePreview').innerHTML = 
            '<img src="<?php echo BASE_URL; ?>assets/img/promotions/' + promo.image + '" style="max-width: 200px; border-radius: 8px;">';
    }
}

function previewImage(input) {
    const preview = document.getElementById('imagePreview');
    preview.innerHTML = '';
    
    if(input.files && input.files[0]) {
        const reader = new FileReader();
        reader.onload = function(e) {
            preview.innerHTML = '<img src="' + e.target.result + '" style="max-width: 200px; border-radius: 8px;">';
        }
        reader.readAsDataURL(input.files[0]);
    }
}

// Close modal on outside click
document.getElementById('promoModal').addEventListener('click', function(e) {
    if(e.target === this) {
        closeModal();
    }
});
</script>

<?php include 'includes/admin-footer.php'; ?>