<?php
$page_title = "Manage Bookings";
include 'includes/admin-header.php';

// Handle status update
if(isset($_POST['update_status'])) {
    $booking_id = (int)$_POST['booking_id'];
    $new_status = mysqli_real_escape_string($conn, $_POST['payment_status']);
    
    $update_query = "UPDATE bookings SET payment_status = ? WHERE id = ?";
    $stmt = mysqli_prepare($conn, $update_query);
    mysqli_stmt_bind_param($stmt, "si", $new_status, $booking_id);
    
    if(mysqli_stmt_execute($stmt)) {
        $_SESSION['success'] = "Booking status updated successfully!";
    } else {
        $_SESSION['error'] = "Failed to update booking status.";
    }
    
    header("Location: bookings.php");
    exit();
}

// Handle delete
if(isset($_GET['delete'])) {
    $booking_id = (int)$_GET['delete'];
    
    $delete_query = "DELETE FROM bookings WHERE id = ?";
    $stmt = mysqli_prepare($conn, $delete_query);
    mysqli_stmt_bind_param($stmt, "i", $booking_id);
    
    if(mysqli_stmt_execute($stmt)) {
        $_SESSION['success'] = "Booking deleted successfully!";
    } else {
        $_SESSION['error'] = "Failed to delete booking.";
    }
    
    header("Location: bookings.php");
    exit();
}

// Get filter parameters
$status_filter = isset($_GET['status']) ? $_GET['status'] : '';
$search = isset($_GET['search']) ? mysqli_real_escape_string($conn, $_GET['search']) : '';

// Build query
$query = "SELECT b.*, u.full_name, u.email, 
          s1.station_name as origin_name, s2.station_name as destination_name,
          t.train_name, sc.departure_time, sc.arrival_time
          FROM bookings b 
          JOIN users u ON b.user_id = u.id 
          JOIN schedules sc ON b.schedule_id = sc.id
          JOIN trains t ON sc.train_id = t.id
          JOIN stations s1 ON sc.origin_station_id = s1.id
          JOIN stations s2 ON sc.destination_station_id = s2.id
          WHERE 1=1";

if($status_filter) {
    $query .= " AND b.payment_status = '$status_filter'";
}

if($search) {
    $query .= " AND (u.full_name LIKE '%$search%' OR u.email LIKE '%$search%' OR b.booking_code LIKE '%$search%')";
}

$query .= " ORDER BY b.created_at DESC";
$bookings = mysqli_query($conn, $query);

include 'includes/admin-sidebar.php';
?>

<div class="admin-container">
    <div class="page-header">
        <div>
            <h1>Manage Bookings</h1>
            <p>View and manage all customer bookings</p>
        </div>
    </div>

    <?php if(isset($_SESSION['success'])): ?>
        <div class="alert-success">
            <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <polyline points="20 6 9 17 4 12"></polyline>
            </svg>
            <?php echo $_SESSION['success']; unset($_SESSION['success']); ?>
        </div>
    <?php endif; ?>

    <?php if(isset($_SESSION['error'])): ?>
        <div class="alert-error">
            <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <circle cx="12" cy="12" r="10"></circle>
                <line x1="15" y1="9" x2="9" y2="15"></line>
                <line x1="9" y1="9" x2="15" y2="15"></line>
            </svg>
            <?php echo $_SESSION['error']; unset($_SESSION['error']); ?>
        </div>
    <?php endif; ?>

    <!-- Filters -->
    <div class="filters-bar">
        <form method="GET" class="filter-form">
            <div class="filter-group">
                <label>Search</label>
                <input type="text" name="search" placeholder="Search by name, email, or booking code..." 
                       value="<?php echo htmlspecialchars($search); ?>">
            </div>
            
            <div class="filter-group">
                <label>Payment Status</label>
                <select name="status">
                    <option value="">All Status</option>
                    <option value="pending" <?php echo $status_filter == 'pending' ? 'selected' : ''; ?>>Pending</option>
                    <option value="confirmed" <?php echo $status_filter == 'confirmed' ? 'selected' : ''; ?>>Confirmed</option>
                    <option value="cancelled" <?php echo $status_filter == 'cancelled' ? 'selected' : ''; ?>>Cancelled</option>
                </select>
            </div>
            
            <button type="submit" class="btn btn-primary">
                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <circle cx="11" cy="11" r="8"></circle>
                    <path d="m21 21-4.35-4.35"></path>
                </svg>
                Filter
            </button>
            <a href="bookings.php" class="btn btn-outline">Reset</a>
        </form>
    </div>

    <!-- Bookings Table -->
    <div class="admin-card">
        <div class="table-container">
            <table class="data-table">
                <thead>
                    <tr>
                        <th>Booking Code</th>
                        <th>Customer</th>
                        <th>Route</th>
                        <th>Train</th>
                        <th>Travel Date</th>
                        <th>Passengers</th>
                        <th>Total Price</th>
                        <th>Payment Status</th>
                        <th>Booking Status</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if(mysqli_num_rows($bookings) > 0): ?>
                        <?php while($booking = mysqli_fetch_assoc($bookings)): ?>
                            <tr>
                                <td><strong><?php echo htmlspecialchars($booking['booking_code']); ?></strong></td>
                                <td>
                                    <div>
                                        <strong><?php echo htmlspecialchars($booking['full_name']); ?></strong><br>
                                        <small style="color: var(--gray);"><?php echo htmlspecialchars($booking['email']); ?></small>
                                    </div>
                                </td>
                                <td>
                                    <small><?php echo htmlspecialchars($booking['origin_name']); ?> → <?php echo htmlspecialchars($booking['destination_name']); ?></small>
                                </td>
                                <td><?php echo htmlspecialchars($booking['train_name']); ?></td>
                                <td><?php echo date('d M Y', strtotime($booking['travel_date'])); ?></td>
                                <td><?php echo $booking['passengers']; ?> seat(s)</td>
                                <td><strong>Rp <?php echo number_format($booking['total_price'], 0, ',', '.'); ?></strong></td>
                                <td>
                                    <form method="POST" style="margin: 0;">
                                        <input type="hidden" name="booking_id" value="<?php echo $booking['id']; ?>">
                                        <select name="payment_status" class="status-badge status-<?php echo $booking['payment_status']; ?>" 
                                                onchange="this.form.submit()" style="border: none; cursor: pointer;">
                                            <option value="pending" <?php echo $booking['payment_status'] == 'pending' ? 'selected' : ''; ?>>Pending</option>
                                            <option value="confirmed" <?php echo $booking['payment_status'] == 'confirmed' ? 'selected' : ''; ?>>Confirmed</option>
                                            <option value="cancelled" <?php echo $booking['payment_status'] == 'cancelled' ? 'selected' : ''; ?>>Cancelled</option>
                                        </select>
                                        <input type="hidden" name="update_status" value="1">
                                    </form>
                                </td>
                                <td>
                                    <span class="status-badge status-<?php echo $booking['booking_status']; ?>">
                                        <?php echo ucfirst($booking['booking_status']); ?>
                                    </span>
                                </td>
                                <td>
                                    <div class="table-actions">
                                        <a href="?delete=<?php echo $booking['id']; ?>" 
                                           class="btn btn-danger btn-sm btn-icon" 
                                           data-confirm="Are you sure you want to delete this booking?"
                                           title="Delete">
                                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                                <polyline points="3 6 5 6 21 6"></polyline>
                                                <path d="M19 6v14a2 2 0 0 1-2 2H7a2 2 0 0 1-2-2V6m3 0V4a2 2 0 0 1 2-2h4a2 2 0 0 1 2 2v2"></path>
                                            </svg>
                                        </a>
                                    </div>
                                </td>
                            </tr>
                        <?php endwhile; ?>
                    <?php else: ?>
                        <tr>
                            <td colspan="10" style="text-align: center; padding: 60px 20px;">
                                <div class="empty-state">
                                    <svg width="80" height="80" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                        <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"></path>
                                        <polyline points="14 2 14 8 20 8"></polyline>
                                    </svg>
                                    <h3>No bookings found</h3>
                                    <p>Try adjusting your filters or search terms</p>
                                </div>
                            </td>
                        </tr>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>

<?php include 'includes/admin-footer.php'; ?>